package ru.yandex.stockpile.server.data.names;

import java.util.Arrays;
import java.util.List;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.kikimr.client.kv.StringMicroUtils;
import ru.yandex.kikimr.util.NameRange;
import ru.yandex.stockpile.server.SnapshotLevel;
import ru.yandex.stockpile.server.data.chunk.SnapshotAddress;
import ru.yandex.stockpile.server.data.names.file.ChunkFile;
import ru.yandex.stockpile.server.data.names.file.CommandFile;
import ru.yandex.stockpile.server.data.names.file.IndexFile;
import ru.yandex.stockpile.server.data.names.file.LogFile;
import ru.yandex.stockpile.server.data.names.parserFormatter.ParserFormatter;
import ru.yandex.stockpile.server.data.names.parserFormatter.ParserFormatters;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class StockpileKvNames {
    public static final String TMP_PREFIX = "t.";
    public static final String LOG_PREFIX = "l.";

    public static final ParserFormatter<Void> tmpPrefixPf = ParserFormatters.str(TMP_PREFIX);

    public static final String CURRENT_PREFIX = "c.";

    public static final String CURRENT_LOG_PREFIX = CURRENT_PREFIX + LOG_PREFIX;

    public static final String BACKUP_FIRST_LETTER = "b";

    public static final NumberNamePart txnPart = new NumberNamePart(17);
    public static final NumberNamePart chunkNoPart = new NumberNamePart(5);

    public static NameRange currentFilesRange() {
        return StringMicroUtils.asciiPrefixToRange(CURRENT_PREFIX);
    }

    public static NameRange logRange() {
        return StringMicroUtils.asciiPrefixToRange(CURRENT_LOG_PREFIX);
    }

    @Nonnull
    public static String chunksPrefixCurrent(SnapshotAddress snapshotAddress) {
        return ChunkFile.chunkPrefixCurrentPf.format(snapshotAddress);
    }

    @Nonnull
    public static String indexPrefixCurrent(SnapshotAddress snapshotAddress) {
        return IndexFile.indexPrefixCurrentDotPf.format(snapshotAddress);
    }

    @Nonnull
    public static String commandPrefixCurrent(SnapshotAddress snapshotAddress) {
        return CommandFile.currentSnapshotDotTxDotPf.format(snapshotAddress);
    }

    public static List<NameRange> currentSnapshot(SnapshotAddress address) {
        return currentSnapshot("", address);
    }

    public static List<NameRange> currentSnapshot(String prefix, SnapshotAddress address) {
        return Arrays.asList(
                StringMicroUtils.asciiPrefixToRange(prefix + StockpileKvNames.indexPrefixCurrent(address)),
                StringMicroUtils.asciiPrefixToRange(prefix + StockpileKvNames.chunksPrefixCurrent(address)),
                StringMicroUtils.asciiPrefixToRange(prefix + StockpileKvNames.commandPrefixCurrent(address))
        );
    }

    public static List<NameRange> currentSnapshot(SnapshotLevel level) {
        return Arrays.asList(
                StringMicroUtils.asciiPrefixToRange(level.index.currentPrefix()),
                StringMicroUtils.asciiPrefixToRange(level.chunk.currentPrefix()),
                StringMicroUtils.asciiPrefixToRange(level.command.currentPrefix())
        );
    }

    public static String chunkFileName(SnapshotLevel level, long snapshotTxn, int chunkNo, FileNamePrefix prefix) {
        return new ChunkFile(level, snapshotTxn, chunkNo).reconstruct(prefix);
    }

    public static NameRange logsToRange(long toTxn, boolean includeTo) {
        if (includeTo) {
            return NameRange.all()
                .ge(CURRENT_LOG_PREFIX)
                .le(LogFile.upperLimitForTxn(toTxn).reconstructCurrent());
        } else {
            return NameRange.all()
                .ge(CURRENT_LOG_PREFIX)
                .lt(LogFile.pfCurrentWithoutPart.format(toTxn));
        }
    }

    public static NameRange logNameRangeInclusive(long fromTxn, long toTxn) {
        if (fromTxn > toTxn) {
            throw new IllegalArgumentException();
        }

        return NameRange.all()
            .ge(LogFile.pfCurrentWithoutPart.format(fromTxn))
            .le(LogFile.upperLimitForTxn(toTxn).reconstructCurrent());
    }

    public static String shortLogName(long txn) {
        return new LogFile(txn, 0, true).reconstructCurrent();
    }

}
