package ru.yandex.stockpile.server.data.names.file;

import javax.annotation.Nonnull;

import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.stockpile.server.SnapshotLevel;
import ru.yandex.stockpile.server.data.chunk.SnapshotAddress;
import ru.yandex.stockpile.server.data.names.FileKind;
import ru.yandex.stockpile.server.data.names.FileNameParsed;
import ru.yandex.stockpile.server.data.names.FileNamePrefix;
import ru.yandex.stockpile.server.data.names.HasTxn;
import ru.yandex.stockpile.server.data.names.StockpileKvNames;
import ru.yandex.stockpile.server.data.names.parserFormatter.ParserFormatter;
import ru.yandex.stockpile.server.data.names.parserFormatter.ParserFormatters;
import ru.yandex.stockpile.server.data.names.parserFormatter.TwoWayFunction;

/**
 * @author Vladimir Gordiychuk
 */
public class ChunkFile extends FileNameParsed implements HasTxn {
    // s<snapshotLevel>.<txn>
    private static final ParserFormatter<SnapshotAddress> chunkPrefixPf =
            ParserFormatters.seq2(SnapshotLevel.chunkPrefix0ParserFormatter(), StockpileKvNames.txnPart.parserFormatter())
                    .addSuffix(".")
                    .map(new TwoWayFunction<>() {
                        @Override
                        public SnapshotAddress thatWay(Tuple2<SnapshotLevel, Long> t) {
                            return new SnapshotAddress(t._1, t._2);
                        }

                        @Override
                        public Tuple2<SnapshotLevel, Long> backwards(SnapshotAddress chunkPrefix) {
                            return Tuple2.tuple(chunkPrefix.level(), chunkPrefix.txn());
                        }
                    });

    // c.s<snapshotLevel>.<txn>
    public static final ParserFormatter<SnapshotAddress> chunkPrefixCurrentPf =
            chunkPrefixPf.addPrefix(FileNamePrefix.Current.pfUnit);

    // s<snapshotLevel>.<txn>.<part><hasMore>
    public static final ParserFormatter<ChunkFile> pf = ParserFormatters.seq2(
            chunkPrefixPf,
            StockpileKvNames.chunkNoPart.integerParserFormatter()
    )
            .map(new TwoWayFunction<>() {
                @Override
                public ChunkFile thatWay(Tuple2<SnapshotAddress, Integer> t) {
                    return new ChunkFile(t._1.level(), t._1.txn(), t._2);
                }

                @Override
                public Tuple2<SnapshotAddress, Integer> backwards(ChunkFile chunk) {
                    return Tuple2.tuple(chunk.snapshotAddress(), chunk.chunkNo);
                }
            });

    // c.s<snapshotLevel>.<txn>.<part><hasMore>
    public static final ParserFormatter<ChunkFile> pfCurrent = pf.addPrefix(FileNamePrefix.Current.pfUnit);

    private final SnapshotLevel level;
    private final long txn;
    private final int chunkNo;

    public ChunkFile(SnapshotLevel level, long txn, int chunkNo) {
        this.level = level;
        this.txn = txn;
        this.chunkNo = chunkNo;
    }

    @Nonnull
    public SnapshotAddress snapshotAddress() {
        return new SnapshotAddress(level, txn);
    }

    @Nonnull
    @Override
    public FileKind fileKind() {
        return level.chunk.kind();
    }

    @Nonnull
    @Override
    public String reconstructWithinBackup() {
        return pf.format(this);
    }

    @Override
    public long txn() {
        return txn;
    }

    public int chunkNo() {
        return chunkNo;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        ChunkFile chunk = (ChunkFile) o;
        if (txn != chunk.txn) return false;
        if (chunkNo != chunk.chunkNo) return false;
        return level == chunk.level;
    }

    @Override
    public int hashCode() {
        int result = level.hashCode();
        result = 31 * result + (int) (txn ^ (txn >>> 32));
        result = 31 * result + chunkNo;
        return result;
    }
}
