package ru.yandex.stockpile.server.data.names.file;

import javax.annotation.Nonnull;

import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.stockpile.server.SnapshotLevel;
import ru.yandex.stockpile.server.data.chunk.SnapshotAddress;
import ru.yandex.stockpile.server.data.names.BigFilePart;
import ru.yandex.stockpile.server.data.names.FileKind;
import ru.yandex.stockpile.server.data.names.FileNameParsed;
import ru.yandex.stockpile.server.data.names.FileNamePrefix;
import ru.yandex.stockpile.server.data.names.HasParts;
import ru.yandex.stockpile.server.data.names.HasTxn;
import ru.yandex.stockpile.server.data.names.StockpileKvNames;
import ru.yandex.stockpile.server.data.names.parserFormatter.ParserFormatter;
import ru.yandex.stockpile.server.data.names.parserFormatter.ParserFormatters;
import ru.yandex.stockpile.server.data.names.parserFormatter.TwoWayFunction;

/**
 * @author Vladimir Gordiychuk
 */
public class CommandFile extends FileNameParsed implements HasTxn, HasParts {
    // i<snapshotLevel>.<txn>.
    private static final ParserFormatter<SnapshotAddress> snapshotDotTxPf =
            ParserFormatters.seq2(SnapshotLevel.commandPrefixParserFormatter(), StockpileKvNames.txnPart.parserFormatter())
                    .addSuffix(".")
                    .map(new TwoWayFunction<>() {
                        @Override
                        public SnapshotAddress thatWay(Tuple2<SnapshotLevel, Long> t) {
                            return new SnapshotAddress(t._1, t._2);
                        }

                        @Override
                        public Tuple2<SnapshotLevel, Long> backwards(SnapshotAddress chunkPrefix) {
                            return Tuple2.tuple(chunkPrefix.level(), chunkPrefix.txn());
                        }
                    });

    // c.i<snapshotLevel>.<txn>.
    public static final ParserFormatter<SnapshotAddress> currentSnapshotDotTxDotPf =
            snapshotDotTxPf.addPrefix(FileNamePrefix.Current.pfUnit);

    // i<snapshotLevel>.<txn>.<part><hasMore>
    public static final ParserFormatter<CommandFile> pf = ParserFormatters.seq2(
            snapshotDotTxPf,
            BigFilePart.suffixPf())
            .map(new TwoWayFunction<>() {
                @Override
                public CommandFile thatWay(Tuple2<SnapshotAddress, BigFilePart> t) {
                    return new CommandFile(t._1.level(), t._1.txn(), t._2.getNo(), t._2.isLast());
                }

                @Override
                public Tuple2<SnapshotAddress, BigFilePart> backwards(CommandFile index) {
                    return Tuple2.tuple(index.snapshotAddress(), index.bigFilePart());
                }
            });

    // c.i<snapshotLevel>.<txn>.<part><hasMore>
    public static final ParserFormatter<CommandFile> pfCurrent = pf.addPrefix(FileNamePrefix.Current.pfUnit);

    private final SnapshotLevel level;
    private final long txn;
    private final int partNo;
    private final boolean last;

    public CommandFile(SnapshotLevel level, long txn, int partNo, boolean last) {
        this.level = level;
        this.txn = txn;
        this.partNo = partNo;
        this.last = last;
    }

    @Nonnull
    public SnapshotLevel getLevel() {
        return level;
    }

    @Override
    public int getPartNo() {
        return partNo;
    }

    @Override
    public boolean isLast() {
        return last;
    }

    @Nonnull
    @Override
    public FileKind fileKind() {
        return level.command.kind();
    }

    @Nonnull
    public SnapshotAddress snapshotAddress() {
        return new SnapshotAddress(level, txn);
    }

    @Nonnull
    @Override
    public String reconstructWithinBackup() {
        return pf.format(this);
    }

    @Override
    public long txn() {
        return txn;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        CommandFile index = (CommandFile) o;
        if (txn != index.txn) return false;
        if (partNo != index.partNo) return false;
        if (last != index.last) return false;
        return level == index.level;
    }

    @Override
    public int hashCode() {
        int result = level.hashCode();
        result = 31 * result + (int) (txn ^ (txn >>> 32));
        result = 31 * result + partNo;
        result = 31 * result + (last ? 1 : 0);
        return result;
    }
}
