package ru.yandex.stockpile.server.data.names.file;

import javax.annotation.Nonnull;

import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.stockpile.server.data.names.BigFilePart;
import ru.yandex.stockpile.server.data.names.FileKind;
import ru.yandex.stockpile.server.data.names.FileNameParsed;
import ru.yandex.stockpile.server.data.names.FileNamePrefix;
import ru.yandex.stockpile.server.data.names.HasParts;
import ru.yandex.stockpile.server.data.names.HasTxn;
import ru.yandex.stockpile.server.data.names.StockpileKvNames;
import ru.yandex.stockpile.server.data.names.parserFormatter.ParserFormatter;
import ru.yandex.stockpile.server.data.names.parserFormatter.ParserFormatters;
import ru.yandex.stockpile.server.data.names.parserFormatter.TwoWayFunction;

/**
 * @author Vladimir Gordiychuk
 */
public class LogFile extends FileNameParsed implements HasTxn, HasParts {
    // l.<txn>
    private static final ParserFormatter<Long> pfWithoutPart =
            StockpileKvNames.txnPart.parserFormatter()
                    .addPrefix(StockpileKvNames.LOG_PREFIX);

    // l.<txn>.<chunkNo>y|z
    public static final ParserFormatter<LogFile> pf = ParserFormatters.seq2(
            pfWithoutPart,
            BigFilePart.dotSuffixPf()
    )
            .map(new TwoWayFunction<>() {
                @Override
                public LogFile thatWay(Tuple2<Long, BigFilePart> t) {
                    return new LogFile(t._1, t._2.getNo(), t._2.isLast());
                }

                @Override
                public Tuple2<Long, BigFilePart> backwards(LogFile log) {
                    return Tuple2.tuple(log.txn, new BigFilePart(log.partNo, log.last));
                }
            });

    // c.l.<txn>.<chunkNo>y|z
    public static final ParserFormatter<LogFile> pfCurrent = pf.addPrefix(FileNamePrefix.Current.pfUnit);

    // c.l.<txn>
    public static final ParserFormatter<Long> pfCurrentWithoutPart =
            pfWithoutPart.addPrefix(FileNamePrefix.Current.pfUnit);

    private final long txn;
    private final int partNo;
    private final boolean last;

    public LogFile(long txn, int partNo, boolean last) {
        this.txn = txn;
        this.partNo = partNo;
        this.last = last;
    }

    public static LogFile upperLimitForTxn(long txn) {
        return new LogFile(txn, BigFilePart.max.getNo(), true);
    }

    @Nonnull
    @Override
    public FileKind fileKind() {
        return FileKind.LOG;
    }

    @Nonnull
    @Override
    public String reconstructWithinBackup() {
        return pf.format(this);
    }

    @Override
    public long txn() {
        return txn;
    }

    public long txn0zForTest() {
        if (partNo != 0 || !last) {
            throw new IllegalStateException();
        }
        return txn;
    }

    @Override
    public int getPartNo() {
        return partNo;
    }

    @Override
    public boolean isLast() {
        return last;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        LogFile log = (LogFile) o;
        if (txn != log.txn) return false;
        if (partNo != log.partNo) return false;
        return last == log.last;
    }

    @Override
    public int hashCode() {
        int result = (int) (txn ^ (txn >>> 32));
        result = 31 * result + partNo;
        result = 31 * result + (last ? 1 : 0);
        return result;
    }
}
