package ru.yandex.stockpile.server.data.names.parserFormatter;

import java.util.Optional;
import java.util.function.BiFunction;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public interface Parser<A> {

    @SafeVarargs
    static <A> Parser<A> or(Parser<? extends A>... parsers) {
        return string -> {
            for (Parser<? extends A> parser : parsers) {
                Result<A> result = (Result<A>) parser.parsePrefix(string);
                if (result instanceof Success<?>) {
                    return result;
                }
            }

            return new Failure<>();
        };
    }

    abstract class Result<A> {
        private Result() {
        }

        public abstract <B> Result<B> thenParse(BiFunction<A, String, Result<B>> f);

        public abstract <B> Result<B> map(Function<A, B> f);

        public abstract Optional<A> full();

        public Success<A> success() {
            return (Success<A>) this;
        }
    }

    class Success<A> extends Result<A> {
        private final A a;
        private final String remaining;

        public Success(A a, String remaining) {
            this.a = a;
            this.remaining = remaining;
        }

        public A prefix() {
            return a;
        }

        @Override
        public <B> Result<B> thenParse(BiFunction<A, String, Result<B>> f) {
            return f.apply(this.a, this.remaining);
        }

        @Override
        public <B> Result<B> map(Function<A, B> f) {
            return new Success<>(f.apply(a), remaining);
        }

        @Override
        public Optional<A> full() {
            return remaining.isEmpty() ? Optional.of(a) : Optional.empty();
        }
    }

    class Failure<A> extends Result<A> {
        @Override
        public <B> Result<B> map(Function<A, B> f) {
            return new Failure<>();
        }

        @Override
        public Optional<A> full() {
            return Optional.empty();
        }

        @Override
        public <B> Result<B> thenParse(BiFunction<A, String, Result<B>> f) {
            return new Failure<>();
        }
    }


    Result<A> parsePrefix(String string);

    default Optional<A> parse(String string) {
        return parsePrefix(string).full();
    }

    default A parseOrThrow(String string) {
        return parse(string).orElseThrow(() -> new RuntimeException("failed to parse string: '" + string + "'"));
    }

    default ParserFormatter<A> asParserFormatterMaybeThrowOnFormat() {
        return ParserAsParserFormatter.of(this);
    }

    default Parser<A> addPrefix(String prefix) {
        return addPrefix(ParserFormatters.str(prefix));
    }

    default Parser<A> addPrefix(ParserFormatter<Void> prefix) {
        return asParserFormatterMaybeThrowOnFormat().addPrefix(prefix);
    }

}
