package ru.yandex.stockpile.server.shard;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ThreadLocalRandom;

import ru.yandex.solomon.memory.layout.MemoryCounter;
import ru.yandex.stockpile.api.EStockpileStatusCode;
import ru.yandex.stockpile.api.grpc.StockpileRuntimeException;
import ru.yandex.stockpile.client.shard.StockpileLocalId;
import ru.yandex.stockpile.server.shard.actor.InActor;

/**
 * @author Vladimir Gordiychuk
 */
public class AllocateLocalIdsProcess extends ShardProcess {
    final StockpileShard shard;
    private final StockpileShardMetrics.AllocateIdsMetrics metrics;
    private final StockpileRequestsQueue<AllocateRequest> queue;

    public AllocateLocalIdsProcess(StockpileShard shard) {
        super(shard, ProcessType.ALLOCATE_LOCAL_IDS, "");
        this.shard = shard;
        this.metrics = shard.metrics.allocateIds;
        this.queue = new StockpileRequestsQueue<>(
            metrics.queueBytes,
            metrics.queueRequests,
            shard.globals.stockpileShardAggregatedStats.allocateLocalIdsInQueueTimeHistogram);
    }

    @Override
    void start(InActor a) {
    }

    @Override
    void act(InActor a) {
        var requests = queue.dequeueAll();
        if (requests.isEmpty()) {
            return;
        }

        // TODO: gordiychuk replace on seq generate SOLOMON-5254
        long now = System.currentTimeMillis();
        for (var req : requests) {
            shard.commonExecutor.execute(() -> {
                if (req.deadline != 0 && req.deadline <= now) {
                    req.doneFuture.completeExceptionally(new StockpileRuntimeException(EStockpileStatusCode.DEADLINE_EXCEEDED, false));
                }
                var random = ThreadLocalRandom.current();
                long[] result = new long[req.size];
                for (int index = 0; index < result.length; index++) {
                    result[index] = StockpileLocalId.random(random);
                }
                req.doneFuture.complete(result);
            });
        }
    }

    public CompletableFuture<long[]> allocateLocalIds(int size, long deadline) {
        var req = new AllocateRequest(size, deadline);
        this.queue.enqueue(req);
        shard.actorRunner.schedule();
        return req.doneFuture;
    }

    @Override
    public long memorySizeIncludingSelf() {
        return queue.memorySizeIncludingSelf();
    }

    @Override
    protected void stoppedReleaseResources() {
        Throwable e = shard.stopException();
        for (var request : queue.dequeueAll()) {
            request.doneFuture.completeExceptionally(e);
        }
    }

    private static class AllocateRequest implements StockpileRequest {
        private static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(AllocateRequest.class)
            + MemoryCounter.CompletableFuture_SELF_SIZE;

        final int size;
        final long deadline;
        final CompletableFuture<long[]> doneFuture;
        final long createdAt;

        public AllocateRequest(int size, long deadline) {
            this.size = size;
            this.deadline = deadline;
            this.doneFuture = new CompletableFuture<>();
            this.createdAt = System.nanoTime();
        }

        @Override
        public long memorySizeIncludingSelf() {
            return SELF_SIZE;
        }

        @Override
        public long getCreatedAtNanos() {
            return createdAt;
        }
    }
}
