package ru.yandex.stockpile.server.shard;

import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.solomon.codec.archive.MetricArchiveGeneric;
import ru.yandex.solomon.codec.archive.MetricArchiveImmutable;
import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.codec.archive.header.DecimPolicyField;
import ru.yandex.solomon.codec.archive.header.DeleteBeforeField;
import ru.yandex.solomon.codec.archive.header.MetricHeader;
import ru.yandex.solomon.codec.archive.header.MetricTypeField;
import ru.yandex.solomon.codec.serializer.OwnerField;
import ru.yandex.solomon.model.point.RecyclableAggrPoint;
import ru.yandex.solomon.model.point.column.StockpileColumns;
import ru.yandex.solomon.model.point.column.TsColumn;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.timeseries.AggrGraphDataListIterator;
import ru.yandex.solomon.model.timeseries.MetricTypeTransfers;
import ru.yandex.stockpile.client.shard.StockpileMetricId;
import ru.yandex.stockpile.server.shard.merge.AggrPointIteratorWrapper;
import ru.yandex.stockpile.server.shard.merge.ArchiveItemIterator;
import ru.yandex.stockpile.server.shard.merge.ConcatIterator;
import ru.yandex.stockpile.server.shard.merge.DeleteBeforeIterator;
import ru.yandex.stockpile.server.shard.merge.EmptyIterator;
import ru.yandex.stockpile.server.shard.merge.Iterator;
import ru.yandex.stockpile.server.shard.merge.MergeIterator;
import ru.yandex.stockpile.server.shard.merge.OneItemIterator;

import static java.util.stream.Collectors.collectingAndThen;
import static java.util.stream.Collectors.toList;

/**
 * @author Vladimir Gordiychuk
 */
public class ArchiveCombiner {
    private static final Logger logger = LoggerFactory.getLogger(ArchiveCombiner.class);

    private final int shardId;
    private final long localId;
    private List<Item> items;

    public ArchiveCombiner(int shardId, long localId) {
        this.shardId = shardId;
        this.localId = localId;
        this.items = new ArrayList<>();
    }

    public void add(MetricArchiveMutable archive) {
        archive.sortAndMerge();
        items.add(new Item(archive));
    }

    public void add(MetricArchiveImmutable archive, long lastTsMillis) {
        items.add(new Item(archive, firstTsMillis(archive), lastTsMillis));
    }

    public CombineResult combine() {
        if (items.isEmpty()) {
            return CombineResult.EMPTY;
        } else if (items.size() == 1) {
            var item = items.get(0);
            return new CombineResult(item.header(), item.recordCount, item.bytesCount, item.iterator);
        }

        items = processDelete(items);
        items = combineNearKinds(items);
        items = convertMetricTypes(shardId, localId, items);

        Item result = new Item(items);
        return new CombineResult(result.header(), result.recordCount, result.bytesCount, result.iterator);
    }

    public static CombineResult combineArchives(int shardId, long localId, List<MetricArchiveImmutable> archives, long[] lastTssMillis) {
        ArchiveCombiner combiner = new ArchiveCombiner(shardId, localId);
        for (int index = 0; index < archives.size(); index++) {
            combiner.add(archives.get(index), lastTssMillis[index]);
        }
        return combiner.combine();
    }

    private static long firstTsMillis(MetricArchiveGeneric archive) {
        var point = RecyclableAggrPoint.newInstance();
        try {
            return archive.iterator().next(point) ? point.tsMillis : 0;
        } finally {
            point.recycle();
        }
    }

    private static List<Item> processDelete(List<Item> items) {
        long maxDeleteBeforeMillis = 0;
        for (int index = items.size() - 1; index > 0; index--) {
            Item item = items.get(index);
            if (item.deleteBefore == DeleteBeforeField.KEEP) {
                continue;
            }

            if (item.deleteBefore == DeleteBeforeField.DELETE_ALL) {
                List<Item> deleteView = items.subList(0, index);
                for (Item delete : deleteView) {
                    item.ownerProjectId = OwnerField.mergeOwnerField(delete.ownerProjectId, item.ownerProjectId);
                    item.ownerShardId = OwnerField.mergeOwnerField(delete.ownerShardId, item.ownerShardId);
                    item.decimPolicyId = DecimPolicyField.merge(delete.decimPolicyId, item.decimPolicyId);
                    item.type = MetricTypeField.merge(delete.type, item.type);
                    item.recordCount += delete.recordCount;
                    item.bytesCount += delete.bytesCount;
                }
                deleteView.clear();
                return items;
            }

            if (maxDeleteBeforeMillis >= item.deleteBefore) {
                continue;
            }

            maxDeleteBeforeMillis = item.deleteBefore;
            for (Item delete : items.subList(0, index)) {
                delete.iterator = DeleteBeforeIterator.of(maxDeleteBeforeMillis, delete.iterator);
            }
        }

        return items;
    }

    private static List<Item> combineNearKinds(List<Item> items) {
        if (items.size() <= 1) {
            return items;
        }

        List<Item> result = new ArrayList<>(items.size());
        MetricType type = items.get(0).type;
        int from = 0;
        for (int to = 1; to < items.size(); to++) {
            Item item = items.get(to);
            if (item.type == MetricType.METRIC_TYPE_UNSPECIFIED) {
                continue;
            }

            if (type == MetricType.METRIC_TYPE_UNSPECIFIED) {
                type = item.type;
                continue;
            }

            if (type != item.type) {
                List<Item> sameKinds = items.subList(from, to);
                if (!sameKinds.isEmpty()) {
                    result.add(new Item(sameKinds));
                }

                type = item.type;
                from = to;
            }
        }

        result.add(new Item(items.subList(from, items.size())));
        return result;
    }

    private static List<Item> convertMetricTypes(int shardId, long localId, List<Item> items) {
        if (items.size() <= 1) {
            return items;
        }

        MetricType type = defineFinalMetricType(items);
        boolean integrateRequired = false;
        for (Item item : items) {
            if (item.recordCount == 0 || item.iterator.columnSetMask() == 0) {
                item.type = type;
                item.iterator = new EmptyIterator(StockpileColumns.minColumnSet(type));
                continue;
            }

            if (!MetricTypeTransfers.isAvailableTransfer(item.type, type)) {
                logger.warn("Not able to convert metric {} type from {} to {}, drop points {}",
                        StockpileMetricId.toString(shardId, localId),
                        item.type,
                        type,
                        item.recordCount);

                item.type = type;
                item.iterator = new EmptyIterator(StockpileColumns.minColumnSet(type));
                continue;
            }

            var converted = MetricTypeTransfers.of(item.type, type, new AggrPointIteratorWrapper(item.iterator));
            item.iterator = OneItemIterator.of(converted, type, item.firstTsMillis, item.lastTsMillis, item.bytesCount);
            item.type = type;
            integrateRequired = true;
        }

        if (integrateRequired) {
            return combineIntegratedArchives(items);
        }

        return items;
    }

    private static List<Item> combineIntegratedArchives(List<Item> items) {
        if (items.size() <= 1) {
            return items;
        }

        if (items.get(items.size() - 1).type != MetricType.RATE) {
            return items;
        }

        // Timeseries was split by multiple archives, archives with the same metric type
        // was already combined into one item, now for rate metrics we should add last value
        // to avoid reset integrate result and as a result lost point on archives boarder
        Item result = new Item();
        List<AggrGraphDataListIterator> iterators = new ArrayList<>(items.size());
        for (Item item : items) {
            result.deleteBefore = DeleteBeforeField.merge(result.deleteBefore, item.deleteBefore);
            result.ownerProjectId = OwnerField.mergeOwnerField(result.ownerProjectId, item.ownerProjectId);
            result.ownerShardId = OwnerField.mergeOwnerField(result.ownerShardId, item.ownerShardId);
            result.decimPolicyId = DecimPolicyField.merge(result.decimPolicyId, item.decimPolicyId);
            result.type = MetricTypeField.merge(result.type, item.type);
            result.recordCount += item.recordCount;
            result.bytesCount += item.bytesCount;
            result.lastTsMillis = Math.max(result.lastTsMillis, item.lastTsMillis);
            result.firstTsMillis = minTs(result.firstTsMillis, item.firstTsMillis);

            iterators.add(new AggrPointIteratorWrapper(item.iterator));
        }

        MetricArchiveMutable archive = new MetricArchiveMutable();
        archive.setType(result.type);
        long lastArchiveValue = 0;
        long lastTsMillis = TsColumn.DEFAULT_VALUE;
        var temp = RecyclableAggrPoint.newInstance();

        itLoop:
        for (AggrGraphDataListIterator it : iterators) {
            if (!it.next(temp)) {
                continue;
            }

            if (temp.tsMillis == lastTsMillis) {
                // aggregate value at particular time can be split on more than one archive
                // to avoid spike we ad as is aggregate values to prev archive
                while (temp.tsMillis == lastTsMillis) {
                    archive.addRecordData(it.columnSetMask(), temp);
                    if (!it.next(temp)) {
                        continue itLoop;
                    }
                }
            } else {
                // avoid add last archive value it archives boarder already looks like counter
                if (Long.compareUnsigned(lastArchiveValue, temp.longValue) <= 0) {
                    lastArchiveValue = 0;
                }
            }

            long lastValue;
            do {
                temp.longValue += lastArchiveValue;
                lastValue = temp.longValue;
                lastTsMillis = temp.tsMillis;
                archive.addRecordData(it.columnSetMask(), temp);
            } while (it.next(temp));
            lastArchiveValue = lastValue;
        }
        temp.recycle();
        archive.sortAndMerge();
        result.iterator = ArchiveItemIterator.of(archive);
        return Collections.singletonList(result);
    }

    private static MetricType defineFinalMetricType(List<Item> items) {
        int fromIndex = 0;
        MetricType from = MetricType.METRIC_TYPE_UNSPECIFIED;
        for (; fromIndex < items.size(); fromIndex++) {
            Item item = items.get(fromIndex);
            from = MetricTypeField.merge(from, item.type);
            if (from != MetricType.METRIC_TYPE_UNSPECIFIED && item.recordCount > 0) {
                break;
            }
        }

        for (int toIndex = items.size() - 1; toIndex > fromIndex; toIndex--) {
            Item item = items.get(toIndex);
            MetricType to = item.type;
            if (MetricTypeTransfers.isAvailableTransfer(from, to)) {
                return to;
            }
        }

        return from;
    }

    public static class CombineResult {
        private static final CombineResult EMPTY = new CombineResult(
                MetricHeader.defaultValue,
                0,
                0,
                EmptyIterator.INSTANCE);

        private final MetricHeader header;
        private final int recordCount;
        private final int elapsedBytes;
        private final Iterator it;

        CombineResult(MetricHeader header, int recordCount, int elapsedBytes, Iterator it) {
            this.header = header;
            this.recordCount = recordCount;
            this.elapsedBytes = elapsedBytes;
            this.it = it;
        }

        public MetricHeader getHeader() {
            return header;
        }

        public int getRecordCount() {
            return recordCount;
        }

        public int getElapsedBytes() {
            return elapsedBytes;
        }

        public AggrGraphDataListIterator getIt() {
            return new AggrPointIteratorWrapper(it);
        }

        public Iterator getItemIterator() {
            return it;
        }
    }

    private static long minTs(long one, long two) {
        if (one == 0) {
            return two;
        } else if (two == 0) {
            return one;
        }

        return Math.min(one, two);
    }

    private static class Item {
        private long deleteBefore = DeleteBeforeField.KEEP;
        private int ownerProjectId;
        private int ownerShardId;
        private int decimPolicyId;
        private MetricType type = MetricType.METRIC_TYPE_UNSPECIFIED;
        private Iterator iterator;
        private int recordCount;
        private int bytesCount;
        private long firstTsMillis;
        private long lastTsMillis;

        Item() {

        }

        Item(MetricArchiveImmutable archive, long firstTsMillis, long lastTsMillis) {
            init(archive, firstTsMillis, lastTsMillis);
            this.iterator = ArchiveItemIterator.of(archive, lastTsMillis);
        }

        Item(MetricArchiveMutable archive) {
            init(archive, archive.getFirstTsMillis(), archive.getLastTsMillis());
            this.iterator = ArchiveItemIterator.of(archive);
        }

        private void init(MetricArchiveGeneric archive, long firstTsMillis, long lastTsMillis) {
            this.deleteBefore = archive.getDeleteBefore();
            this.ownerProjectId = archive.getOwnerProjectId();
            this.ownerShardId = archive.getOwnerShardId();
            this.decimPolicyId = archive.getDecimPolicyId();
            this.type = archive.getType();
            this.recordCount = archive.getRecordCount();
            this.bytesCount = archive.bytesCount();
            this.firstTsMillis = firstTsMillis;
            this.lastTsMillis = lastTsMillis;
        }

        Item(List<Item> items) {
            for (Item item : items) {
                deleteBefore = DeleteBeforeField.merge(deleteBefore, item.deleteBefore);
                ownerProjectId = OwnerField.mergeOwnerField(ownerProjectId, item.ownerProjectId);
                ownerShardId = OwnerField.mergeOwnerField(ownerShardId, item.ownerShardId);
                decimPolicyId = DecimPolicyField.merge(decimPolicyId, item.decimPolicyId);
                type = MetricTypeField.merge(type, item.type);
                recordCount += item.recordCount;
                bytesCount += item.bytesCount;
                lastTsMillis = Math.max(lastTsMillis, item.lastTsMillis);
                firstTsMillis = minTs(firstTsMillis, item.firstTsMillis);
            }

            iterator = combinePoints(items);
        }

        private Iterator combinePoints(List<Item> items) {
            if (items.size() == 1) {
                return items.get(0).iterator;
            }

            items = items.stream()
                .filter(item -> item.recordCount > 0)
                .collect(Collectors.toList());

            List<Iterator> overlaps = new ArrayList<>(items.size());
            int from = 0;
            for (int to = 1; to < items.size(); to++) {
                if (items.get(to).firstTsMillis > items.get(to - 1).lastTsMillis) {
                    continue;
                }

                if (to - from == 1) {
                    overlaps.add(items.get(from).iterator);
                } else {
                    overlaps.add(concat(items, from, to));
                }
                from = to;
            }

            overlaps.add(concat(items, from, items.size()));
            return MergeIterator.of(overlaps);
        }

        private Iterator concat(List<Item> items, int from, int to) {
            return items.subList(from, to)
                .stream()
                .filter(item -> item.recordCount > 0)
                .map(item -> item.iterator)
                .collect(collectingAndThen(toList(), ConcatIterator::of));
        }

        public MetricHeader header() {
            return new MetricHeader(deleteBefore, ownerProjectId, ownerShardId, decimPolicyId, type);
        }

        @Override
        public String toString() {
            return "Item{" +
                "firstTsMillis=" + Instant.ofEpochMilli(firstTsMillis) +
                ", lastTsMillis=" + Instant.ofEpochMilli(lastTsMillis) +
                '}';
        }
    }
}
