package ru.yandex.stockpile.server.shard;

import java.time.Instant;
import java.util.Objects;
import java.util.concurrent.TimeUnit;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNullableByDefault;

import com.fasterxml.jackson.annotation.JsonAutoDetect;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.annotation.JsonSerialize;

/**
 * @author Vladimir Gordiychuk
 */
@JsonAutoDetect(
    setterVisibility = JsonAutoDetect.Visibility.NONE,
    getterVisibility = JsonAutoDetect.Visibility.NONE,
    fieldVisibility = JsonAutoDetect.Visibility.PUBLIC_ONLY,
    isGetterVisibility = JsonAutoDetect.Visibility.NONE,
    creatorVisibility = JsonAutoDetect.Visibility.NONE
)
@JsonSerialize
@JsonIgnoreProperties(ignoreUnknown = true)
@ParametersAreNullableByDefault
public class KvLock {
    public static final String FILE = ".lock";

    @JsonProperty("fqdn")
    public String host;
    @JsonProperty("pid")
    public long pid;
    @JsonProperty("expiredAt")
    public long expiredAtSeconds;

    public KvLock() {
    }

    public KvLock(String host, long pid, long expiredAtSeconds) {
        this.host = host;
        this.pid = pid;
        this.expiredAtSeconds = expiredAtSeconds;
    }

    public static KvLock parse(@Nonnull byte[] bytes) {
        try {
            ObjectMapper mapper = new ObjectMapper();
            return mapper.readValue(bytes, KvLock.class);
        } catch (Throwable e) {
            throw new RuntimeException("Unable to parse lock content " + new String(bytes), e);
        }
    }

    public static byte[] serialize(KvLock lock) {
        try {
            ObjectMapper mapper = new ObjectMapper();
            return mapper.writeValueAsBytes(lock);
        } catch (Throwable e) {
            throw new RuntimeException("unable serialize " + lock, e);
        }
    }

    public boolean isExpired() {
        long now = TimeUnit.MILLISECONDS.toSeconds(System.currentTimeMillis());
        return now > expiredAtSeconds;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        KvLock lock = (KvLock) o;

        if (pid != lock.pid) return false;
        if (expiredAtSeconds != lock.expiredAtSeconds) return false;
        return Objects.equals(host, lock.host);
    }

    @Override
    public int hashCode() {
        int result = host != null ? host.hashCode() : 0;
        result = 31 * result + (int) (pid ^ (pid >>> 32));
        result = 31 * result + (int) (expiredAtSeconds ^ (expiredAtSeconds >>> 32));
        return result;
    }

    @Override
    public String toString() {
        return String.format(".lock{fqdn=\"%s\", pid=\"%s\", expiredAt=\"%s\"}", host, pid, Instant.ofEpochSecond(expiredAtSeconds));
    }
}
