package ru.yandex.stockpile.server.shard;

import javax.annotation.Nullable;

import it.unimi.dsi.fastutil.longs.AbstractLongComparator;
import it.unimi.dsi.fastutil.longs.Long2ObjectOpenHashMap;
import it.unimi.dsi.fastutil.longs.LongArrays;

import ru.yandex.solomon.codec.archive.MetricArchiveImmutable;
import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.stockpile.client.shard.StockpileLocalId;
import ru.yandex.stockpile.server.data.chunk.ChunkWithNo;
import ru.yandex.stockpile.server.data.chunk.ChunkWriter;
import ru.yandex.stockpile.server.data.index.SnapshotIndexContent;

/**
 * @author Vladimir Gordiychuk
 */
public class LazySnapshotBuilder {
    private final Long2ObjectOpenHashMap<MetricArchiveMutable> data;
    private final long[] sortedKeys;
    private final ChunkWriter writer;
    @Nullable
    private SnapshotIndexContent index;
    private int keyIdx;

    public LazySnapshotBuilder(SnapshotReason reason, Long2ObjectOpenHashMap<MetricArchiveMutable> data) {
        this.writer = new ChunkWriter(reason, System.currentTimeMillis(), 0);
        this.data = data;
        long[] keys = data.keySet().toLongArray();
        LongArrays.quickSort(keys, new AbstractLongComparator() {
            @Override
            public int compare(long a, long b) {
                return StockpileLocalId.compare(a, b);
            }
        });
        this.sortedKeys = keys;
    }

    @Nullable
    public ChunkWithNo nextChunk() {
        if (index != null) {
            return null;
        }

        ChunkWithNo chunk;
        while (keyIdx < sortedKeys.length) {
            long localId = sortedKeys[keyIdx++];
            var source = data.get(localId);
            if (source == null) {
                throw new NullPointerException("Null archive by localId: " + StockpileLocalId.toString(localId));
            }

            if (source.isSortedAndMerged()) {
                chunk = addMetric(localId, source);
            } else {
                try (var copy = new MetricArchiveMutable(source)) {
                    chunk = addMetric(localId, copy);
                }
            }

            if (chunk != null) {
                return chunk;
            }
        }

        var result = writer.finish();
        index = result.index;
        return result.chunkWithNo;
    }

    public SnapshotIndexContent index() {
        if (index == null) {
            throw new NullPointerException("index not build yet, pos "+ keyIdx + " from " + sortedKeys.length);
        }
        return index;
    }

    @Nullable
    private ChunkWithNo addMetric(long localId, MetricArchiveMutable archive) {
        // close frame force merge and sort
        archive.closeFrame();
        return addMetric(localId, archive.getLastTsMillis(), archive.toImmutableNoCopy());
    }

    @Nullable
    private ChunkWithNo addMetric(long localId, long latestTsMillis, MetricArchiveImmutable archive) {
        return writer.writeMetric(localId, latestTsMillis, archive);
    }
}
