package ru.yandex.stockpile.server.shard;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.List;
import java.util.LongSummaryStatistics;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.OptionalInt;
import java.util.OptionalLong;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.ByteString;
import it.unimi.dsi.fastutil.ints.Int2LongMap;
import it.unimi.dsi.fastutil.ints.Int2LongOpenHashMap;
import it.unimi.dsi.fastutil.longs.Long2ObjectMap;

import ru.yandex.kikimr.client.kv.KikimrKvClient;
import ru.yandex.kikimr.client.kv.KvReadRangeResult;
import ru.yandex.kikimr.util.NameRange;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.misc.lang.EnumMapUtils;
import ru.yandex.solomon.codec.archive.MetricArchiveImmutable;
import ru.yandex.solomon.codec.serializer.ByteStringsStockpile;
import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.memory.layout.MemoryBySubsystem;
import ru.yandex.solomon.memory.layout.MemoryCounter;
import ru.yandex.solomon.staffOnly.manager.special.InstantMillis;
import ru.yandex.solomon.util.ExceptionUtils;
import ru.yandex.stockpile.kikimrKv.counting.ReadClass;
import ru.yandex.stockpile.memState.LogEntriesContent;
import ru.yandex.stockpile.memState.MetricToArchiveMap;
import ru.yandex.stockpile.server.SnapshotLevel;
import ru.yandex.stockpile.server.data.DeletedShardSet;
import ru.yandex.stockpile.server.data.chunk.SnapshotAddress;
import ru.yandex.stockpile.server.data.index.SnapshotIndex;
import ru.yandex.stockpile.server.data.index.SnapshotIndexContent;
import ru.yandex.stockpile.server.data.index.SnapshotIndexContentSerializer;
import ru.yandex.stockpile.server.data.log.StockpileProducerSeqNoSnapshotSerializer;
import ru.yandex.stockpile.server.data.names.FileKind;
import ru.yandex.stockpile.server.data.names.FileNameParsed;
import ru.yandex.stockpile.server.data.names.FileNamePrefix;
import ru.yandex.stockpile.server.data.names.InBackupFileNameSet;
import ru.yandex.stockpile.server.data.names.IndexAddressWithFileCount;
import ru.yandex.stockpile.server.data.names.LogAddressWithFileCount;
import ru.yandex.stockpile.server.data.names.StockpileKvNames;
import ru.yandex.stockpile.server.data.names.file.CommandFile;
import ru.yandex.stockpile.server.data.names.file.IndexFile;
import ru.yandex.stockpile.server.data.names.file.ProducerSeqNoFile;
import ru.yandex.stockpile.server.shard.actor.ActorRunnableType;
import ru.yandex.stockpile.server.shard.actor.InActor;
import ru.yandex.stockpile.server.shard.load.Async;
import ru.yandex.stockpile.server.shard.load.AsyncFnIterator;
import ru.yandex.stockpile.server.shard.load.KvLogParsingIterator;
import ru.yandex.stockpile.server.shard.load.KvReadRangeIterator;
import ru.yandex.stockpile.server.shard.load.MetricArchivesMerger;
import ru.yandex.stockpile.server.shard.stat.LevelSizeAndCount;
import ru.yandex.stockpile.server.shard.stat.SizeAndCount;
import ru.yandex.stockpile.server.shard.stat.StockpileShardDiskStats;

import static ru.yandex.stockpile.server.shard.ExceptionHandler.isGenerationChanged;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
class LoadProcess extends ShardProcess {
    public static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(LoadProcess.class);

    @InstantMillis
    private final long started = System.currentTimeMillis();

    private List<KikimrKvClient.KvEntryStats> list;
    private InBackupFileNameSet inBackupFileNameSet;

    private Int2LongMap producerSeqNoById;

    @Nullable
    private volatile AllIndexes indexes;

    @Nullable
    private StockpileShardDiskStats diskStats;
    private volatile double progress;


    public LoadProcess(StockpileShard shard) {
        super(shard, ProcessType.LOAD, "");
        shard.stateCheckAndSet(s -> s instanceof StockpileShardStateInit, new LoadStateImpl(shard));
    }

    @Override
    protected void stoppedReleaseResources() {
        AllIndexes indexes = this.indexes;
        if (indexes != null) {
            indexes.destroy();
        }
    }

    @Override
    public void start(InActor a) {
        incrementGeneration()
            .thenCompose(unit -> listFiles())
            .thenCompose(unit -> deleteTempFiles())
            .thenCompose(unit -> initIndexList())
            .thenCompose(unit -> loadIndexes())
            .thenCompose(unit -> loadProducerSeqNoSnapshot())
            .thenCompose(unit -> newLogsLoad())
            .handle(this::initDone)
            .thenCompose(f -> f)
            .whenComplete((unit, throwable) -> {
                if (throwable != null) {
                    shard.error(throwable);
                    if (isGenerationChanged(throwable)) {
                        generationChanged();
                    } else {
                        RuntimeException e = new RuntimeException("failed to load shard: " + shard.shardId, throwable);
                        ExceptionUtils.uncaughtException(e);
                    }
                }
                shard.actorRunner.schedule();
            });
    }

    public double getLoadingProgress() {
        return progress;
    }

    private CompletableFuture<Void> incrementGeneration() {
        return loopUntilSuccessFuture("incrementGeneration", shard.storage::lock);
    }

    class LoadStateImpl extends StockpileShardState {
        public LoadStateImpl(StockpileShard shard) {
            super(shard);
        }

        @Override
        protected Stream<SnapshotIndexWithStats> indexesWithStatsUnsafe() {
            AllIndexes indexes2 = LoadProcess.this.indexes;
            return indexes2 != null ? indexes2.streamWithStats() : Stream.empty();
        }

        @Override
        public StockpileShard.LoadState loadStateForMon() {
            return shard.lastError != null ? StockpileShard.LoadState.LOADING_ERRORS : StockpileShard.LoadState.LOADING;
        }

        @Override
        public String loadStateDescForMon() {
            return loadStateForMon() + ": at: " + "just started";
        }

        @Override
        public StockpileShardDiskStats diskStats() {
            return Optional.ofNullable(diskStats).orElse(StockpileShardDiskStats.zeroes());
        }

        @Override
        public OptionalInt snapshotCount(SnapshotLevel level) {
            return OptionalInt.empty();
        }

        @Override
        public OptionalLong recordCountReliable() {
            AllIndexes indexes = LoadProcess.this.indexes;
            if (indexes != null) {
                return OptionalLong.of(indexes.recordCount());
            } else {
                return OptionalLong.empty();
            }
        }

        @Override
        public OptionalLong metricsCountReliable() {
            AllIndexes indexes = LoadProcess.this.indexes;
            if (indexes != null) {
                return OptionalLong.of(indexes.metricCount());
            } else {
                return OptionalLong.empty();
            }
        }

        @Override
        public void stop(InActor a) {
            AllIndexes copy = LoadProcess.this.indexes;
            if (copy != null) {
                copy.destroy();
            }
        }

        @Override
        public void addMemoryBySubsystem(MemoryBySubsystem memory) {
            AllIndexes copy = LoadProcess.this.indexes;
            if (copy != null) {
                memory.addAllMemory(copy.memoryBySystem());
            }
        }
    }

    private CompletableFuture<Void> listFiles() {
        CompletableFuture<List<KikimrKvClient.KvEntryStats>> listCurrentFuture =
            loopUntilSuccessFuture("readRangeNames", () -> {
                return shard.storage.readRangeNames(ReadClass.START_READ_LOG, StockpileKvNames.currentFilesRange());
            });

        return listCurrentFuture.thenApply(listCurrent -> {
            updateDiskStats(listCurrent);

            this.list = listCurrent;

            List<IndexFile> indexes = listCurrent.stream()
                .map(e -> IndexFile.pfCurrent.parse(e.getName()).orElse(null))
                .filter(Objects::nonNull)
                .collect(Collectors.toList());

            // consistency check
            IndexAddressWithFileCount.fold(indexes);
            return null;
        });
    }

    private void updateDiskStats(List<KikimrKvClient.KvEntryStats> list) {
        EnumMap<FileKind, SizeAndCount> s = EnumMapUtils.consts(FileKind.class, new SizeAndCount(0, 0));
        for (KikimrKvClient.KvEntryStats e : list) {
            if (!e.getName().startsWith(StockpileKvNames.CURRENT_PREFIX)) {
                continue;
            }

            s.compute(
                FileKind.classify(e.getName()),
                (k, v) -> SizeAndCount.plus(v, new SizeAndCount(e.getSize(), 1)));
        }
        diskStats = new StockpileShardDiskStats(s);
    }


    private CompletableFuture<Void> initIndexList() {
        List<FileNameParsed> currentParsed = list.stream()
            .filter(e -> e.getName().startsWith(StockpileKvNames.CURRENT_PREFIX))
            .map(e -> FileNameParsed.parseCurrent(e.getName()))
            .collect(Collectors.toList());

        this.inBackupFileNameSet = new InBackupFileNameSet(currentParsed);

        long lastUsedTxnFromList = this.inBackupFileNameSet.lastTxn();
        CompletableFuture<Void> future = new CompletableFuture<>();
        shard.runAsync(ActorRunnableType.MISC, a -> {
            log(formatLogPrefix(" now ready to accept writes, last used txn " + lastUsedTxnFromList));
            shard.txTracker = new TxTracker(lastUsedTxnFromList);
            future.complete(null);
        }).whenComplete((ignore, e) -> {
            if (e != null) {
                future.completeExceptionally(e);
            } else {
                future.complete(null);
            }
        });
        return future;
    }

    private CompletableFuture<Void> deleteTempFiles() {
        if (shard.txTracker != null) {
            return CompletableFuture.failedFuture(new IllegalStateException("Not available remove temp files after initialize TxTracker"));
        }

        return loopUntilSuccessFuture("deleteTmpFiles", shard.storage::deleteTempFiles);
    }

    /**
     * Read iterator with retries and counting total files size.
     */
    private final class ShardReadRangeIterator extends KvReadRangeIterator {

        ShardReadRangeIterator(NameRange nameRange) {
            super(nameRange);
        }

        @Override
        protected CompletableFuture<KvReadRangeResult> readNext(NameRange nameRange) {
            return loopUntilSuccessFuture("readRange", () -> shard.storage.readRange(ReadClass.START_READ_LOG, nameRange));
        }
    }

    private CompletableFuture<Void> loadProducerSeqNoSnapshot() {
        if (!StockpileFormat.CURRENT.ge(StockpileFormat.IDEMPOTENT_WRITE_38)) {
            producerSeqNoById = new Int2LongOpenHashMap();
            return CompletableFuture.completedFuture(null);
        }
        return loopUntilSuccessFuture("loadProducerSeqNoSnapshot", () -> {
            return shard.storage.readData(ReadClass.OTHER, ProducerSeqNoFile.CURRENT_FILE_NAME);
        }).thenAccept(optional -> {
            producerSeqNoById = optional
                    .map(StockpileProducerSeqNoSnapshotSerializer.S::deserializeFull)
                    .orElseGet(Int2LongOpenHashMap::new);
        });
    }

    private CompletableFuture<Void> newLogsLoad() {
        //
        // It is better to use system's common FJ pool here, because:
        //   1) using dedicated thread pool here is wasteful, since load process is temporary
        //      and after it finishes dedicated threads will only consume memory without profit
        //
        //   2) sharing CpuLowPriority thread pool is not good, because already loaded shards
        //      will start fight with not yet loaded shard for the pool queue
        //
        //   3) FJ pool will start more threads when they are needed and after load is done
        //      it will stop not used anymore threads
        //
        ExecutorService executor = ForkJoinPool.commonPool();

        final int totalCount = inBackupFileNameSet.getLogs().size();
        final AtomicInteger loadedCount = new AtomicInteger(0);

        LongSummaryStatistics statistics = inBackupFileNameSet.getLogs()
            .stream()
            .mapToLong(LogAddressWithFileCount::getTxn)
            .summaryStatistics();

        CompletableFuture<LogEntriesContent> doneFuture;
        if (statistics.getCount() == 0) {
            doneFuture = CompletableFuture.completedFuture(new LogEntriesContent(new MetricToArchiveMap(), new DeletedShardSet(), producerSeqNoById, 0, 0));
        } else {
            NameRange range = StockpileKvNames.logNameRangeInclusive(statistics.getMin(), statistics.getMax());
            ShardReadRangeIterator readIt = new ShardReadRangeIterator(range);
            var invalidArchiveStrategy = shard.globals.invalidArchiveStrategy;
            KvLogParsingIterator parseIt = new KvLogParsingIterator(executor, invalidArchiveStrategy.strategy, readIt);

            MetricArchivesMerger merger = new MetricArchivesMerger(shard.shardId, invalidArchiveStrategy, executor, Runtime.getRuntime().availableProcessors());
            DeletedShardSet deletedShards = new DeletedShardSet();
            doneFuture = Async.forEachAsync(parseIt, logEntry -> {
                producerSeqNoById.putAll(logEntry.getProducerSeqNoById());
                deletedShards.addAll(logEntry.getDeletedShards());
                Long2ObjectMap<MetricArchiveImmutable> metrics = logEntry.getDataByMetricId();
                double progress = loadedCount.incrementAndGet() * 100. / totalCount;
                log("loaded " + String.format("%.2f%%", progress));
                this.progress = progress;
                if (metrics.isEmpty()) {
                    return CompletableFuture.completedFuture(null);
                } else {
                    return merger.merge(metrics);
                }
            }).thenApply(aVoid -> {
                MetricToArchiveMap metricsMap = merger.combineResults();
                long totalSize = parseIt.getTotalSize();
                return new LogEntriesContent(metricsMap, deletedShards, producerSeqNoById, loadedCount.get(), totalSize);
            });
        }

        return doneFuture.thenAccept(logEntries -> {
            log("logs load done");
            shard.run(ActorRunnableType.DONE_LOGS_LOAD, a -> {
                // must be in actor thread, writes are in progress
                shard.logState.logsLoaded(logEntries);
            });
        });
    }

    private CompletableFuture<SnapshotIndexWithStats> loadIndexFile(IndexAddressWithFileCount indexName, SizeAndCount commandSize) {
        List<CompletableFuture<Optional<byte[]>>> partsFutures = new ArrayList<>(indexName.getCount());

        for (IndexFile index : indexName.indexFileNames()) {
            String fileName = index.reconstructCurrent();
            partsFutures.add(loopUntilSuccessFuture(
                "loadIndexes",
                () -> shard.storage.readData(ReadClass.START_READ_INDEX, fileName)
            ));
        }

        return CompletableFutures.allOf(partsFutures)
            .thenApply(parts -> {
                // TODO: optimize it
                ByteString[] bytes = parts.stream()
                    .map(buf -> ByteStringsStockpile.unsafeWrap(buf.get()))
                    .toArray(ByteString[]::new);

                int totalSize = 0;
                for (ByteString b : bytes) {
                    totalSize += b.size();
                }

                SnapshotIndexContent indexContent = SnapshotIndexContentSerializer.S.deserializeParts(bytes);
                SizeAndCount indexSize = new SizeAndCount(totalSize, bytes.length);
                SizeAndCount chunkSize = indexContent.diskSize();
                LevelSizeAndCount levelSize = new LevelSizeAndCount(indexSize, chunkSize, commandSize);
                SnapshotIndex index = new SnapshotIndex(indexName.getLevel(), indexName.getTxn(), indexContent);
                return new SnapshotIndexWithStats(index, levelSize);
            });
    }

    private CompletableFuture<Void> loadIndexes() {
        var commandDiskSizeByAddress = commandDiskSize();
        var indexes = inBackupFileNameSet.getIndexes();
        var it = new AsyncFnIterator<>(indexes.iterator(),
                index -> loadIndexFile(index, commandDiskSizeByAddress.getOrDefault(index.snapshotAddress(), SizeAndCount.zero)));
        var loaded = new ArrayList<SnapshotIndexWithStats>(indexes.size());
        return Async.forEach(it, loaded::add)
                .thenRunAsync(() -> loadIndexDone(loaded), shard.commonExecutor);
    }

    private Map<SnapshotAddress, SizeAndCount> commandDiskSize() {
        Map<SnapshotAddress, SizeAndCount> result = new HashMap<>();
        for (var file : list) {
            var command = CommandFile.pfCurrent.parse(file.getName()).orElse(null);
            if (command == null) {
                continue;
            }

            var disSize = new SizeAndCount(file.getSize(), 1);
            var address = command.snapshotAddress();
            var prev = result.put(address, disSize);
            if (prev != null) {
                result.put(address, SizeAndCount.plus(prev, disSize));
            }
        }
        return result;
    }

    private void loadIndexDone(List<SnapshotIndexWithStats> indexes) {
        Map<SnapshotLevel, List<SnapshotIndexWithStats>> indexesByLevel = indexes.stream()
            .collect(Collectors.groupingBy(i -> i.getIndex().getLevel()));

        List<SnapshotIndexWithStats> twoHourIndexes = indexesByLevel.getOrDefault(SnapshotLevel.TWO_HOURS, List.of());
        List<SnapshotIndexWithStats> dailyIndexes = indexesByLevel.getOrDefault(SnapshotLevel.DAILY, List.of());
        List<SnapshotIndexWithStats> eternityIndexes = indexesByLevel.getOrDefault(SnapshotLevel.ETERNITY, List.of());
        if (this.indexes != null) {
            throw new IllegalStateException("indexes already loaded");
        }

        var allIndexes = new AllIndexes(shard.debug, twoHourIndexes, dailyIndexes, eternityIndexes);
        this.indexes = allIndexes;
        shard.memoryLimit += allIndexes.memorySizeIncludingSelf();
        shard.globals.updateShardMemoryLimits.updateShardMemoryLimits();

        Set<String> files = list.stream()
            .map(KikimrKvClient.KvEntryStats::getName)
            .collect(Collectors.toSet());

        List<String> missingFiles = this.indexes.stream()
            .flatMap(i -> Arrays.stream(i.chunkAddresses()))
            .map(a -> StockpileKvNames.chunkFileName(a.getLevel(), a.getSnapshotTxn(), a.getChunkNo(), FileNamePrefix.Current.instance))
            .filter(n -> !files.contains(n))
            .collect(Collectors.toList());

        if (!missingFiles.isEmpty()) {
            throw new RuntimeException("missing files: " + missingFiles);
        }

        // TODO: report snapshots without indexes
    }

    private long durationToNow() {
        return System.currentTimeMillis() - started;
    }

    private CompletableFuture<Void> initDone(Void aVoid, @Nullable Throwable t) {
        if (isGenerationChanged(t)) {
            generationChanged();
            return CompletableFuture.completedFuture(null);
        }

        return shard.runAsync(ActorRunnableType.DONE_INIT, a -> {
            if (t != null) {
                throw new RuntimeException("shard init failed", t);
            }

            AllIndexes indexCopy = Objects.requireNonNull(indexes);
            StockpileShardStateDone shardStateDone = new StockpileShardStateDone(shard, indexCopy);
            indexes = null;
            shard.stateCheckAndSet(s -> s instanceof LoadProcess.LoadStateImpl, shardStateDone);
            shard.stats.loadDurationMillis = durationToNow();
            shard.loadProcess = null;
            shard.initializedOrError.countDown();
            completedSuccessfullyWriteStats();
        });
    }

    @Override
    public long memorySizeIncludingSelf() {
        return SELF_SIZE;
    }
}
