package ru.yandex.stockpile.server.shard;

import java.util.ArrayList;
import java.util.List;

import ru.yandex.stockpile.api.EStockpileStatusCode;
import ru.yandex.stockpile.api.grpc.StockpileRuntimeException;
import ru.yandex.stockpile.server.Txn;
import ru.yandex.stockpile.server.shard.actor.ActorRunnableType;
import ru.yandex.stockpile.server.shard.actor.InActor;

/**
 * @author Vladimir Gordiychuk
 */
public class LogProcess extends ShardProcess {
    private static final int MAX_INFLIGHT = 1;
    private static final StockpileRuntimeException SHARD_NOT_READY =
        new StockpileShardStateException(EStockpileStatusCode.SHARD_NOT_READY, "cannot process writes yet");
    final StockpileShard shard;
    private final StockpileShardMetrics.WriteMetrics metrics;
    private final StockpileRequestsQueue<StockpileWriteRequest> writeQueue;

    public LogProcess(StockpileShard shard) {
        super(shard, ProcessType.WRITE, "");
        this.shard = shard;
        this.metrics = shard.metrics.write;
        writeQueue = new StockpileRequestsQueue<>(
            metrics.queueBytes,
            metrics.queueRequests,
            shard.globals.stockpileShardAggregatedStats.writeInQueueTimeHistogram);
    }

    @Override
    protected void stoppedReleaseResources() {
        Throwable e = shard.stopException();
        for (var request : writeQueue.dequeueAll()) {
            request.getFuture().completeExceptionally(e);
            request.release();
        }
    }

    @Override
    void start(InActor a) {
    }

    @Override
    void act(InActor a) {
        if (metrics.txInFlight.get() >= MAX_INFLIGHT) {
            return;
        }

        List<StockpileWriteRequest> requests = dequeue();
        if (requests.isEmpty()) {
            return;
        }

        if (shard.isStop()) {
            Throwable e = shard.stopException();
            for (var request : requests) {
                request.getFuture().completeExceptionally(e);
                request.release();
            }
            return;
        }

        if (!shard.canServeWrites()) {
            for (var request : requests) {
                request.getFuture().completeExceptionally(SHARD_NOT_READY);
                request.release();
            }
            return;
        }

        write(a, requests);
    }

    private List<StockpileWriteRequest> dequeue() {
        List<StockpileWriteRequest> requests = writeQueue.dequeueAll();
        if (requests.isEmpty()) {
            return requests;
        }

        List<StockpileWriteRequest> filtered = new ArrayList<>(requests.size());
        List<StockpileWriteRequest> obsolete = new ArrayList<>();
        for (var request : requests) {
            if (isObsolete(request)) {
                log(" ignore request producerId="
                        + request.getProducerId()
                        + ", producerSeqNo=" + request.getProducerSeqNo()
                        + ", actualProducerSeqNo=" + getProducerSeqNo(request.getProducerId())
                        + ", records=" + request.getRecordsInArchives());
                obsolete.add(request);
            } else {
                filtered.add(request);
            }
        }
        if (!obsolete.isEmpty()) {
            shard.metrics.write.ignore(obsolete);
            shard.commonExecutor.execute(() -> {
                Txn txn = new Txn(1);
                for (var req : requests) {
                    req.getFuture().complete(txn);
                }
            });
        }
        return filtered;
    }

    private boolean isObsolete(StockpileWriteRequest request) {
        return shard.logState.isObsolete(request.getProducerId(), request.getProducerSeqNo());
    }

    public long getProducerSeqNo(int producerId) {
        return shard.logState.getProducerSeqNo(producerId);
    }

    public void enqueue(StockpileWriteRequest request) {
        if (shard.isMemoryLimitReach()) {
            shard.run(ActorRunnableType.MISC, shard::checkSizeStartSnapshot);
            long usage = shard.memoryUsage();
            long overcommit = usage - shard.memoryLimit;
            String message = "Shard " + shard.shardId + " memory limit: " + shard.memoryLimit + " used " + usage + " overcommit " + overcommit;
            request.release();
            StockpileRuntimeException e = new StockpileRuntimeException(EStockpileStatusCode.RESOURCE_EXHAUSTED, message, false);
            request.getFuture().completeExceptionally(e);
            return;
        }
        writeQueue.enqueue(request);
    }

    private void write(InActor actor, List<StockpileWriteRequest> requests) {
        var tx = new TxWrite(this, shard.txTracker.allocateTx(), requests);
        metrics.txInFlight.add(1);
        tx.start(actor);
    }

    void written(TxWrite write) {
        metrics.txInFlight.add(-1);
        shard.run(ActorRunnableType.COMPLETE_WRITE, a -> {
            shard.txTracker.written(write);
            shard.checkSizeStartSnapshot(a);
        });
    }

    void failed(TxWrite write, Throwable e) {
        metrics.txInFlight.add(-1);
        shard.run(ActorRunnableType.COMPLETE_WRITE, a -> {
            shard.txTracker.written(new TxNoop(write.txn()));
        });
    }

    @Override
    public long memorySizeIncludingSelf() {
        return writeQueue.memorySizeIncludingSelf();
    }
}
