package ru.yandex.stockpile.server.shard;

import java.util.Arrays;
import java.util.Objects;
import java.util.OptionalLong;
import java.util.stream.Stream;

import javax.annotation.ParametersAreNonnullByDefault;

import it.unimi.dsi.fastutil.ints.Int2LongMap;
import it.unimi.dsi.fastutil.ints.Int2LongOpenHashMap;
import it.unimi.dsi.fastutil.longs.LongOpenHashSet;
import it.unimi.dsi.fastutil.longs.LongSet;
import org.apache.commons.lang3.StringUtils;

import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.memory.layout.MemMeasurableSubsystem;
import ru.yandex.solomon.memory.layout.MemoryBySubsystem;
import ru.yandex.solomon.memory.layout.MemoryCounter;
import ru.yandex.solomon.staffOnly.html.HtmlWriterWithCommonLibraries;
import ru.yandex.solomon.staffOnly.manager.ExtraContentParam;
import ru.yandex.solomon.staffOnly.manager.special.ExtraContent;
import ru.yandex.stockpile.memState.LogEntriesContent;
import ru.yandex.stockpile.memState.MetricToArchiveMap;
import ru.yandex.stockpile.server.data.DeletedShardSet;
import ru.yandex.stockpile.server.data.log.LogReason;
import ru.yandex.stockpile.server.data.log.StockpileLogEntryContent;
import ru.yandex.stockpile.server.shard.stat.SizeAndCount;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class LogState implements MemMeasurableSubsystem, AutoCloseable {
    private static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(LogState.class);

    private final int shardId;
    private LogEntriesContent contentInLogSnapshots = new LogEntriesContent();
    private LogEntriesContent contentSinceLastLogSnapshot = new LogEntriesContent();
    private long firstTxnSinceLastLogSnapshot = 0;
    private volatile boolean forceLogSnapshot = false;

    public LogState(int shardId) {
        this.shardId = shardId;
    }

    long[] metricIds() {
        LongSet ids = new LongOpenHashSet(contentInLogSnapshots.getMetricToArchiveMap().getMetrics().keySet());
        ids.addAll(contentSinceLastLogSnapshot.getMetricToArchiveMap().getMetrics().keySet());
        return ids.toLongArray();
    }

    SizeAndCount diskStats() {
        int fileCount = contentInLogSnapshots.getFileCount() + contentSinceLastLogSnapshot.getFileCount();
        long fileSize = contentInLogSnapshots.getFileSize() + contentSinceLastLogSnapshot.getFileSize();
        return new SizeAndCount(fileSize, fileCount);
    }

    public boolean isObsolete(int producerId, long producerSeqNo) {
        if (producerId == 0) {
            return false;
        }

        return getProducerSeqNo(producerId) >= producerSeqNo;
    }

    public long getProducerSeqNo(int producerId) {
        long lastTxProducerSeqNo = contentSinceLastLogSnapshot.getProducerSeqNo(producerId);
        if (lastTxProducerSeqNo != 0) {
            return lastTxProducerSeqNo;
        }

        return contentInLogSnapshots.getProducerSeqNo(producerId);
    }

    public boolean isEmpty() {
        return contentInLogSnapshots.isEmpty() && contentSinceLastLogSnapshot.isEmpty();
    }

    OptionalLong firstTxnSinceLastLogSnapshot() {
        if (firstTxnSinceLastLogSnapshot != 0) {
            return OptionalLong.of(firstTxnSinceLastLogSnapshot);
        } else {
            return OptionalLong.empty();
        }
    }

    Stream<MetricArchiveMutable> contentSinceLastSnapshotStream(long localId) {
        return Stream.of(contentInLogSnapshots, contentSinceLastLogSnapshot)
            .map(content -> content.getMetricToArchiveMap().getById(localId))
            .filter(Objects::nonNull);
    }

    void logsLoaded(LogEntriesContent loaded) {
        var diff = contentInLogSnapshots;
        try {
            appendDelta(loaded, diff.asLogEntryContent(), diff.getFileCount(), diff.getFileSize());
            contentInLogSnapshots = loaded;
        } finally {
            diff.release();
        }
    }

    void completeWriteLogTx(StockpileLogEntryContent mergedLogEntry, long logEntryDiskSize, long txn) {
        if (firstTxnSinceLastLogSnapshot == 0) {
            firstTxnSinceLastLogSnapshot = txn;
        }
        appendDelta(contentSinceLastLogSnapshot, mergedLogEntry, 1, logEntryDiskSize);
    }

    void completeWriteLogSnapshot(long logEntryDiskSize, long txn) {
        contentInLogSnapshots.addSize(1, logEntryDiskSize);
    }

    private void appendDelta(LogEntriesContent target, StockpileLogEntryContent delta, long fileCount, long diskSize) {
        target.updateWithLogEntry(shardId, delta);
        target.addSize(fileCount, diskSize);
    }

    LogEntriesContent takeLogSnapshot() {
        if (contentSinceLastLogSnapshot.isEmpty()) {
            throw new IllegalStateException("first txn since last snapshot: " + firstTxnSinceLastLogSnapshot);
        }

        LogEntriesContent result = contentSinceLastLogSnapshot;
        contentSinceLastLogSnapshot = new LogEntriesContent(contentSinceLastLogSnapshot.getMetricCount());
        firstTxnSinceLastLogSnapshot = 0;
        forceLogSnapshot = false;
        // unknown serialized disk size at this moment
        appendDelta(contentInLogSnapshots, result.asLogEntryContent(), 0, 0);
        return result;
    }

    LogEntriesContent takeForTwoHourSnapshot() {
        LogEntriesContent snapshot = contentInLogSnapshots;
        snapshot.updateWithLogEntriesContent(shardId, contentSinceLastLogSnapshot);

        contentInLogSnapshots = new LogEntriesContent(new MetricToArchiveMap(snapshot.getMetricCount()), new DeletedShardSet(), snapshot.getProducerSeqNoById(), 0, 0);
        contentSinceLastLogSnapshot = new LogEntriesContent(contentSinceLastLogSnapshot.getMetricCount());
        firstTxnSinceLastLogSnapshot = 0;
        return snapshot;
    }

    @Override
    public void addMemoryBySubsystem(MemoryBySubsystem memory) {
        memory.addMemory("stockpile.shard.unflushed.logs", contentSinceLastLogSnapshot.memorySizeIncludingSelf());
        memory.addMemory("stockpile.shard.unflushed.snapshots", contentInLogSnapshots.memorySizeIncludingSelf());
        memory.addMemory("stockpile.shard.unflushed.other", SELF_SIZE);
    }

    @Override
    public long memorySizeIncludingSelf() {
        return SELF_SIZE
                + contentSinceLastLogSnapshot.memorySizeIncludingSelf()
                + contentInLogSnapshots.memorySizeIncludingSelf();
    }

    public long estimateLogSnapshotSize() {
        MetricToArchiveMap metricToArchiveMap = contentSinceLastLogSnapshot.getMetricToArchiveMap();
        return metricToArchiveMap.estimateSerializedSize();
    }

    public LogReason anyReasonToLog() {
        if (firstTxnSinceLastLogSnapshot == 0) {
            // there were no any transactions after last log snapshot
            return LogReason.TX;
        } else if (forceLogSnapshot) {
            return LogReason.LS_EXPLICITLY_REQUESTED;
        } else if (contentSinceLastLogSnapshot.getFileCount() >= LogStateOptions.FILES_COUNT) {
            // there were too many transactions since last log snapshot
            return LogReason.LS_TOO_MANY_LOGS;
        }

        // anyway just write last logs delta
        return LogReason.TX;
    }

    public void requestForceLogSnapshot() {
        forceLogSnapshot = true;
    }

    @Override
    public void close() {
        contentInLogSnapshots.release();
        contentSinceLastLogSnapshot.release();
    }

    @ExtraContent("Producer SeqNo")
    public void producerSeqNoTable(ExtraContentParam p) {
        Int2LongMap seqNoByProducerId = new Int2LongOpenHashMap();
        seqNoByProducerId.putAll(contentInLogSnapshots.getProducerSeqNoById());
        seqNoByProducerId.putAll(contentSinceLastLogSnapshot.getProducerSeqNoById());

        var producerIds = seqNoByProducerId.keySet().toIntArray();
        Arrays.sort(producerIds);

        HtmlWriterWithCommonLibraries hw = p.getHtmlWriter();
        hw.pre(() -> {
            hw.writeRaw(String.format("%15s", "producerId"));
            hw.writeRaw(" ");
            hw.writeRaw(String.format("%7s", "cluster"));
            hw.writeRaw(" ");
            hw.writeRaw(String.format("%7s", "shard"));
            hw.writeRaw(" ");
            hw.writeRaw(String.format("%15s", "seqNo"));
            hw.nl();

            hw.writeRaw(StringUtils.repeat('-', 15));
            hw.writeRaw(" ");
            hw.writeRaw(StringUtils.repeat('-', 7));
            hw.writeRaw(" ");
            hw.writeRaw(StringUtils.repeat('-', 7));
            hw.writeRaw(" ");
            hw.writeRaw(StringUtils.repeat('-', 15));
            hw.nl();

            for (int producerId : producerIds) {
                int cluster = producerId >> 16;
                int shardId = producerId & 0x0000ffff;
                long seqNo = seqNoByProducerId.get(producerId);

                hw.writeRaw(String.format("%15d", producerId));
                hw.writeRaw(" ");
                hw.writeRaw(String.format("%7d", cluster));
                hw.writeRaw(" ");
                hw.writeRaw(String.format("%7d", shardId));
                hw.writeRaw(" ");
                hw.writeRaw(String.format("%15d", seqNo));
                hw.nl();
            }
        });
    }
}
