package ru.yandex.stockpile.server.shard;

import java.time.Clock;
import java.util.Optional;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import ru.yandex.solomon.config.protobuf.stockpile.TMergeConfig;
import ru.yandex.solomon.config.protobuf.stockpile.TMergeOptions;

import static ru.yandex.solomon.config.OptionalSet.setBytesSize;
import static ru.yandex.solomon.config.OptionalSet.setInt;
import static ru.yandex.solomon.config.OptionalSet.setTime;

/**
 * @author Vladimir Gordiychuk
 */
@Configuration
@ParametersAreNonnullByDefault
public class MergeContext {
    private final TMergeConfig config;

    public MergeContext(Optional<TMergeConfig> config) {
        this.config = config.orElse(TMergeConfig.getDefaultInstance());
    }

    @Bean
    public MergeStrategy mergeStrategy(Clock clock) {
        var daily = options(config.getDailyOptions());
        var eternity = options(config.getEternityOptions());

        if (daily.isEnableNew() && daily.getSplitDelayMillis() != 0) {
            eternity = eternity.toBuilder()
                .usePrevLevels(false)
                .build();
        }

        return new MergeStrategy(daily, eternity, clock);
    }

    private static MergeOptions options(TMergeOptions proto) {
        var opts = MergeOptions.newBuilder();
        opts.enableNew(proto.getEnableNew());
        setInt(opts::snapshotsLimit, proto.getSnapshotLimit());
        setBytesSize(opts::minSnapshotBytesSize, proto.getMinSnapshotSize());
        setTime(opts::splitDelayMillis, proto.getSplitDelay());
        setTime(opts::forceMergeAfterMillis, proto.getForceMergeAfter());
        setTime(opts::forceMergeAfterJitterMillis, proto.getForceMergeAfterJitter());
        opts.allowDecim(proto.getAllowDecim());
        return opts.build();
    }
}
