package ru.yandex.stockpile.server.shard;

import java.util.concurrent.TimeUnit;

import javax.annotation.Nullable;

import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.dataSize.DataSizeUnit;
import ru.yandex.solomon.staffOnly.annotations.ManagerMethod;
import ru.yandex.solomon.util.time.DurationUtils;

/**
 * @author Vladimir Gordiychuk
 */
public class MergeOptions {
    private boolean enableNew;
    private int snapshotsLimit;
    private long minSnapshotByteSize;
    private long splitDelayMillis;
    private long forceMergeAfterMillis;
    private long forceMergeAfterJitterMillis;
    private boolean allowDecim;
    private boolean usePrevLevels;

    private MergeOptions(Builder builder) {
        this.enableNew = builder.enableNew != null
            ? builder.enableNew
            : false;
        this.snapshotsLimit = builder.snapshotsLimit != null
            ? builder.snapshotsLimit
            : 3;
        this.minSnapshotByteSize = builder.minSnapshotByteSize != null
            ? builder.minSnapshotByteSize
            : 0;
        this.splitDelayMillis = builder.splitDelayMillis != null
            ? builder.splitDelayMillis
            : 0;
        this.forceMergeAfterMillis = builder.forceMergeAfterMillis != null
            ? builder.forceMergeAfterMillis
            : 0;
        this.forceMergeAfterJitterMillis = builder.forceMergeAfterJitterMillis != null
            ? builder.forceMergeAfterJitterMillis
            : 0;
        this.allowDecim = builder.allowDecim != null
            ? builder.allowDecim
            : false;
        this.usePrevLevels = builder.usePrevLevels != null
            ? builder.usePrevLevels
            : true;
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public Builder toBuilder() {
        return new Builder(this);
    }

    public boolean isEnableNew() {
        return enableNew;
    }

    public int getSnapshotsLimit() {
        return snapshotsLimit;
    }

    public boolean isEnableSnapshotLimit() {
        return snapshotsLimit > 0;
    }

    public long getMinSnapshotBytesSize() {
        return minSnapshotByteSize;
    }

    public long getSplitDelayMillis() {
        return splitDelayMillis;
    }

    public long getForceMergeAfterMillis() {
        return forceMergeAfterMillis;
    }

    public long getForceMergeAfterJitterMillis() {
        return forceMergeAfterJitterMillis;
    }

    public boolean isEnableForceAfterMillis() {
        return forceMergeAfterMillis > 0;
    }

    public boolean isEnableForceAfterJitterMillis() {
        return forceMergeAfterJitterMillis > 0;
    }

    public boolean isEnableMinSnapshotBytesLimit() {
        return minSnapshotByteSize > 0;
    }

    public boolean isAllowDecim() {
        return allowDecim;
    }

    public boolean isUsePrevLevels() {
        return usePrevLevels;
    }

    @ManagerMethod
    private void setEnableNew(boolean enableNew) {
        this.enableNew = enableNew;
    }

    @ManagerMethod
    public void setSnapshotsLimit(int snapshotsLimit) {
        this.snapshotsLimit = snapshotsLimit;
    }

    @ManagerMethod
    private void setMinSnapshotByteSize(long value, DataSizeUnit unit) {
        this.minSnapshotByteSize = unit.dataSize(value).toBytes();
    }

    @ManagerMethod
    public void setSplitDelayMillis(long value, TimeUnit unit) {
        this.splitDelayMillis = unit.toMillis(value);
    }

    @ManagerMethod
    public void setForceMergeAfterMillis(long value, TimeUnit unit) {
        this.forceMergeAfterMillis = unit.toMillis(value);
    }

    @ManagerMethod
    public void setForceMergeAfterJitterMillis(long value, TimeUnit unit) {
        this.forceMergeAfterJitterMillis = unit.toMillis(value);
    }

    @ManagerMethod
    public void setAllowDecim(boolean value) {
        this.allowDecim = value;
    }

    @ManagerMethod
    public void setUsePrevLevels(boolean value) {
        this.usePrevLevels = value;
    }

    @Override
    public String toString() {
        return "MergeOptions{" +
            "enableNew=" + enableNew +
            ", snapshotsLimit=" + snapshotsLimit +
            ", minSnapshotByteSize=" + DataSize.shortString(minSnapshotByteSize) +
            ", splitDelay=" + DurationUtils.formatDurationMillis(splitDelayMillis) +
            ", forceAfterMillis=" + DurationUtils.formatDurationMillis(forceMergeAfterMillis) +
            ", forceAfterJitter=" + DurationUtils.formatDurationMillis(forceMergeAfterJitterMillis) +
            ", allowDecim=" + allowDecim +
            ", usePrevLabels=" + usePrevLevels +
            '}';
    }

    public static class Builder {
        @Nullable
        private Boolean enableNew;
        private Integer snapshotsLimit;
        private Long minSnapshotByteSize;
        private Long splitDelayMillis;
        private Long forceMergeAfterMillis;
        private Long forceMergeAfterJitterMillis;
        private Boolean allowDecim;
        private Boolean usePrevLevels;

        private Builder() {
        }

        private Builder(MergeOptions options) {
            this.enableNew = options.enableNew;
            this.snapshotsLimit = options.snapshotsLimit;
            this.minSnapshotByteSize = options.minSnapshotByteSize;
            this.splitDelayMillis = options.splitDelayMillis;
            this.forceMergeAfterMillis = options.forceMergeAfterMillis;
            this.forceMergeAfterJitterMillis = options.forceMergeAfterJitterMillis;
            this.allowDecim = options.allowDecim;
            this.usePrevLevels = options.usePrevLevels;
        }

        public Builder enableNew(boolean enable) {
            this.enableNew = enable;
            return this;
        }

        public Builder snapshotsLimit(int snapshotsLimit) {
            this.snapshotsLimit = snapshotsLimit;
            return this;
        }

        public Builder minSnapshotBytesSize(int value, DataSizeUnit unit) {
            minSnapshotByteSize = unit.dataSize(value).toBytes();
            return this;
        }

        public Builder minSnapshotBytesSize(long bytes) {
            minSnapshotByteSize = bytes;
            return this;
        }

        public Builder splitDelayMillis(long value, TimeUnit unit) {
            splitDelayMillis = unit.toMillis(value);
            return this;
        }

        public Builder forceMergeAfterMillis(long value, TimeUnit unit) {
            forceMergeAfterMillis = unit.toMillis(value);
            return this;
        }

        public Builder forceMergeAfterJitterMillis(long value, TimeUnit unit) {
            forceMergeAfterJitterMillis = unit.toMillis(value);
            return this;
        }

        public Builder allowDecim(boolean allow) {
            this.allowDecim = allow;
            return this;
        }

        public Builder usePrevLevels(boolean value) {
            this.usePrevLevels = value;
            return this;
        }

        public MergeOptions build() {
            return new MergeOptions(this);
        }
    }
}
