package ru.yandex.stockpile.server.shard;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Flow;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.kikimr.client.kv.KikimrKvGenerationChangedCheckedException;
import ru.yandex.kikimr.client.kv.KikimrKvGenerationChangedRuntimeException;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.memory.layout.MemMeasurable;
import ru.yandex.solomon.memory.layout.MemoryCounter;
import ru.yandex.stockpile.client.shard.StockpileShardId;
import ru.yandex.stockpile.memState.MetricIdAndData;
import ru.yandex.stockpile.server.Txn;
import ru.yandex.stockpile.server.data.DeletedShardSet;
import ru.yandex.stockpile.server.data.chunk.SnapshotAddress;
import ru.yandex.stockpile.server.data.command.SnapshotCommandContent;
import ru.yandex.stockpile.server.data.index.SnapshotIndex;
import ru.yandex.stockpile.server.shard.MergeProcessMetrics.MergeKindMetrics;
import ru.yandex.stockpile.server.shard.actor.ActorRunnable;
import ru.yandex.stockpile.server.shard.actor.ActorRunnableType;
import ru.yandex.stockpile.server.shard.actor.InActor;

import static ru.yandex.misc.concurrent.CompletableFutures.whenComplete;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
class MergeProcess extends ShardProcess {
    private static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(MergeProcess.class);

    public final MergeKind mergeKind;
    private final CompletableFuture<Txn> doneFuture = new CompletableFuture<>();
    private final StockpileShardStateDone shardStateDone;
    private final MergeKindMetrics metrics;

    private final MergeReader reader;
    private final MergeMerger merger;
    private final MergeWriter currentWriter;
    @Nullable
    private final MergeWriter nextWriter;
    private final CompletableFuture<List<Optional<SnapshotIndexWithStats>>> doneWriteFuture;

    private final long now;
    private final long snapshotTxn;
    private final MergeAdvice advice;
    private final long startTimeNanos;

    public MergeProcess(
        StockpileShardStateDone shardStateDone,
        ArrayList<CompletableFuture<Txn>> mergeCompletedFutures,
        MergeKind mergeKind,
        MergeProcessMetrics metrics,
        InActor a)
    {
        super(shardStateDone.shard, mergeKind.processType, "");
        this.shardStateDone = shardStateDone;
        this.mergeKind = mergeKind;
        this.startTimeNanos = System.nanoTime();
        this.metrics = metrics.getMergeKindMetrics(mergeKind);

        if (shard.mergeProcess != null) {
            throw new IllegalStateException("Previous merge process " + shard.mergeProcess.snapshotTxn +" not finished yet: "+ shard.mergeProcess.getProgress());
        }

        if (shard.twoHourSnapshotProcess != null) {
            throw new IllegalStateException("Not able start merge process when two hours snapshot inflight");
        }

        mergeCompletedFutures.forEach(future -> whenComplete(doneFuture, future));
        this.snapshotTxn = shardStateDone.shard.txTracker.allocateCompletedTxn();
        this.advice = shardStateDone.shard.globals.mergeStrategy.chooseMerge(shardStateDone.indexesWithStatsUnsafe(), mergeKind);

        now = shardStateDone.shard.globals.clock.millis();
        long decimatedAt = advice.allowDecim ? now : 0;
        var invalidArchiveStrategy = shardStateDone.shard.globals.invalidArchiveStrategy;
        reader = new MergeReader(this, advice.indexes, this.metrics);
        merger = new MergeMerger(this, now, advice.allowDecim, advice.splitDelayMillis, invalidArchiveStrategy, this.metrics);
        currentWriter = new MergeWriter(this, now, mergeKind.targetLevel, decimatedAt, snapshotTxn, this.metrics);
        if (advice.splitDelayMillis != 0) {
            nextWriter = new MergeWriter(this, now, mergeKind.targetLevel.nextLevel(), decimatedAt, snapshotTxn, this.metrics);
            doneWriteFuture = CompletableFutures.allOf(List.of(currentWriter.getDoneFuture(), nextWriter.getDoneFuture())).thenApply(list -> list);
        } else {
            nextWriter = null;
            doneWriteFuture = currentWriter.getDoneFuture().thenApply(List::of);
        }

        for (SnapshotIndex index : advice.indexes) {
            Txn.validateTxn(index.getTxn());
            if (index.getTxn() >= snapshotTxn) {
                throw new IllegalStateException("txn " + index.getTxn() + " >= " + snapshotTxn);
            }
        }
    }

    @Override
    protected void stoppedReleaseResources() {
        currentWriter.cancel();
        if (nextWriter != null) {
            nextWriter.cancel();
        }
        RuntimeException e = shard.stopException();
        doneFuture.completeExceptionally(e);
    }

    @Override
    public void start(InActor a) {
        mergeCommands()
                .thenAccept(content -> inActor(actor -> start(actor, content)))
                .exceptionally(e -> onError(new RuntimeException(mergeKind + " merge at shard " + StockpileShardId.toString(shard.shardId) + " failed", e)));
    }

    private void start(InActor actor, SnapshotCommandContent command) {
        // current level
        {
            var deleted = filterDeletedShards(command.deletedShards(), currentWriter);
            var level = filterByCurrentLevel(deleted);
            merger.subscribe(level);
        }
        // next level
        if (nextWriter != null) {
            var deleted = filterDeletedShards(command.deletedShards(), nextWriter);
            var level = filterByNextLevel(deleted);
            merger.subscribe(level);
        }

        reader.subscribe(merger);

        doneWriteFuture
                .thenAccept(optionals -> {
                    var result = optionals.stream()
                            .filter(Optional::isPresent)
                            .map(Optional::get)
                            .collect(Collectors.toList());

                    onCompleteWrite(result);
                })
                .exceptionally(e -> onError(new RuntimeException(mergeKind + " merge at shard " + StockpileShardId.toString(shard.shardId) + " failed", e)));
    }

    private Flow.Processor<MergeTaskResult, MetricIdAndData> filterByCurrentLevel(Flow.Subscriber<MetricIdAndData> subscriber) {
        var filter = new MergeFilter(MergeTaskResult::getCurrentLevel, advice.allowDelete);
        filter.subscribe(subscriber);
        return filter;
    }

    private Flow.Processor<MergeTaskResult, MetricIdAndData> filterByNextLevel(Flow.Subscriber<MetricIdAndData> subscriber) {
        var filter = new MergeFilter(MergeTaskResult::getNextLevel, false);
        filter.subscribe(subscriber);
        return filter;
    }

    private Flow.Subscriber<MetricIdAndData> filterDeletedShards(DeletedShardSet deletedShards, Flow.Subscriber<MetricIdAndData> subscriber) {
        if (deletedShards.isEmpty()) {
            return subscriber;
        }

        var filter = new MergeDeletedShardFilter(deletedShards);
        filter.subscribe(subscriber);
        return filter;
    }

    private CompletableFuture<SnapshotCommandContent> mergeCommands() {
        var loader = new SnapshotCommandLoaderImpl(this);
        var merger = new SnapshotCommandMerger(loader);
        var snapshots = Stream.of(advice.indexes)
                .map(index -> new SnapshotAddress(index.getLevel(), index.getTxn()))
                .collect(Collectors.toList());

        return merger.merge(snapshots)
                .thenCompose(content -> {
                    return nextWriter().writeCommand(content, advice.allowDelete).thenApply(ignore -> content);
                });
    }

    private MergeWriter nextWriter() {
        if (nextWriter != null) {
            return nextWriter;
        }

        return currentWriter;
    }

    private <T> T onError(Throwable e) {
        doneFuture.completeExceptionally(e);
        Throwable cause = e;
        while (cause != null) {
            if (cause instanceof KikimrKvGenerationChangedCheckedException) {
                generationChanged();
                return null;
            } else if (cause instanceof KikimrKvGenerationChangedRuntimeException) {
                generationChanged();
                return null;
            }

            cause = cause.getCause();
        }

        if (shard.isStop()) {
            return null;
        }

        shard.run(ActorRunnableType.MISC, a2 -> {
            throw new RuntimeException("merge process is failed", e);
        });

        return null;
    }

    private void onCompleteWrite(List<SnapshotIndexWithStats> results) {
        inActor(a -> {
            shard.setRenameInProgress();
            shard.globals.stockpileShardHacksForTest.pauseBeforeRename();

            // make sure no read requests started after rename start until rename completion
            shard.storage.flushReadQueue()
                    .thenCompose(ignore -> renameIndexes(results.stream()
                            .map(SnapshotIndexWithStats::address)
                            .toArray(SnapshotAddress[]::new)))
                    .thenRun(() -> onCompleteTx(results))
                    .exceptionally(this::onError);
        });
    }

    private void onCompleteTx(List<SnapshotIndexWithStats> indexes) {
        inActor(a -> {
            if (shard.mergeProcess != MergeProcess.this) {
                throw new IllegalStateException();
            }

            for (var index : advice.indexes) {
                shardStateDone.indexes.removeSnapshot(index.snapshotAddress());
            }

            for (var index : indexes) {
                shardStateDone.indexes.addSnapshot(index);
            }
            shardStateDone.indexes.updateRecordCountStats();
            shardStateDone.indexes.updateLatestSnapshotTime(mergeKind.targetLevel, now);

            long elapsedNanos = System.nanoTime() - startTimeNanos;
            completedSuccessfullyWriteStats();
            doneFuture.complete(new Txn(snapshotTxn));
            shard.mergeProcess = null;
            shard.clearRenameInProgress();
            metrics.addTotalTime(elapsedNanos);
            shard.actorRunner.schedule();
        });
    }

    private void inActor(Consumer<InActor> consumer) {
        processRunInActor(new ActorRunnable() {
            @Override
            public ActorRunnableType type() {
                return ActorRunnableType.MISC;
            }

            @Override
            public void run(InActor a) {
                consumer.accept(a);
            }
        });
    }

    @Override
    public long memorySizeIncludingSelf() {
        long size = SELF_SIZE;
        size += MemoryCounter.CompletableFuture_SELF_SIZE;
        size += advice.memorySizeIncludingSelf();
        size += reader.memorySizeIncludingSelf();
        size += merger.memorySizeIncludingSelf();
        size += MemMeasurable.memorySizeOfNullable(currentWriter);
        size += MemMeasurable.memorySizeOfNullable(nextWriter);
        return size;
    }

    private CompletableFuture<?> renameIndexes(SnapshotAddress[] snapshots) {
        var delete = Stream.of(advice.indexes)
            .map(SnapshotIndex::snapshotAddress)
            .toArray(SnapshotAddress[]::new);

        return loopUntilSuccessFuture("rename",
            () -> shard.storage.renameSnapshotDeleteOld(snapshots, delete));
    }

    public double getProgress() {
        long total = Stream.of(advice.indexes)
            .mapToLong(index -> index.getContent().getMetricCount())
            .sum();

        return merger.getMergedMetrics() * 100. / total;
    }
}
