package ru.yandex.stockpile.server.shard;

import java.util.EnumMap;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import org.springframework.stereotype.Component;

import ru.yandex.monlib.metrics.MetricConsumer;
import ru.yandex.monlib.metrics.MetricSupplier;
import ru.yandex.monlib.metrics.histogram.Histograms;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.primitives.Counter;
import ru.yandex.monlib.metrics.primitives.GaugeInt64;
import ru.yandex.monlib.metrics.primitives.Histogram;
import ru.yandex.monlib.metrics.primitives.LazyGaugeInt64;
import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

/**
 * @author Vladimir Gordiychuk
 */
@Component
public class MergeProcessMetrics implements MetricSupplier {
    private final Map<MergeKind, MergeKindMetrics> mergeKindsMetrics;

    public MergeProcessMetrics() {
        Map<MergeKind, MergeKindMetrics> mergeTaskMetrics = new EnumMap<>(MergeKind.class);
        for (MergeKind kind : MergeKind.values()) {
            mergeTaskMetrics.put(kind, new MergeKindMetrics(kind.name()));
        }

        this.mergeKindsMetrics = mergeTaskMetrics;
    }

    public MergeKindMetrics getMergeKindMetrics(MergeKind kind) {
        return mergeKindsMetrics.get(kind);
    }

    @Override
    public int estimateCount() {
        return mergeKindsMetrics.values()
                .stream()
                .mapToInt(MergeKindMetrics::estimateCount)
                .sum();
    }

    @Override
    public void append(long tsMillis, Labels commonLabels, MetricConsumer consumer) {
        MergeKindMetrics total = new MergeKindMetrics("total");
        for (MergeKindMetrics kind : mergeKindsMetrics.values()) {
            total.combine(kind);
            kind.append(tsMillis, commonLabels, consumer);
        }
        total.append(tsMillis, commonLabels, consumer);
    }

    public static class MergeKindMetrics implements MetricSupplier {
        private final MetricRegistry registry;
        private final MergeTaskMetrics taskMetrics;
        private final Histogram readElapsedTime;
        private final Histogram mergeElapsedTime;
        private final Histogram writeElapsedTime;
        private final Histogram totalElapsedTime;
        final Histogram readMaxLag;
        final Histogram mergeMaxLag;
        final GaugeInt64 writeQueue;
        final Counter invalidArchives;

        public MergeKindMetrics(String kind) {
            this.registry = new MetricRegistry(Labels.of("kind", kind));
            this.taskMetrics = new MergeTaskMetrics(kind);

            readElapsedTime = registry.histogramRate("mergeProcess.read.elapsedTimeMillis", Histograms.exponential(15, 2, 8));
            mergeElapsedTime = registry.histogramRate("mergeProcess.merge.elapsedTimeMillis", Histograms.exponential(15, 2, 1));
            writeElapsedTime = registry.histogramRate("mergeProcess.write.elapsedTimeMillis", Histograms.exponential(15, 2, 8));
            totalElapsedTime = registry.histogramRate("mergeProcess.total.elapsedTimeMinutes", Histograms.exponential(14, 2, 1));
            invalidArchives = registry.counter("mergeProcess.invalidArchives");

            readMaxLag = registry.histogramRate("mergeProcess.read.maxLag", Histograms.exponential(14, 2, 1));
            mergeMaxLag = registry.histogramRate("mergeProcess.merge.maxLag", Histograms.exponential(14, 2, 1));
            writeQueue = registry.gaugeInt64("mergeProcess.write.queue");
        }

        public MergeTaskMetrics getTaskMetrics() {
            return taskMetrics;
        }

        public void addReadTime(long elapsedTimeNanos) {
            readElapsedTime.record(TimeUnit.NANOSECONDS.toMillis(elapsedTimeNanos));
        }

        public void addMergeTime(long elapsedTimeNanos) {
            mergeElapsedTime.record(TimeUnit.NANOSECONDS.toMillis(elapsedTimeNanos));
        }

        public void addWriteTime(long elapsedTimeNanos) {
            writeElapsedTime.record(TimeUnit.NANOSECONDS.toMillis(elapsedTimeNanos));
        }

        public void addTotalTime(long elapsedTimeNanos) {
            totalElapsedTime.record(TimeUnit.NANOSECONDS.toMinutes(elapsedTimeNanos));
        }

        public void combine(MergeKindMetrics metrics) {
            taskMetrics.combine(metrics.taskMetrics);
            readElapsedTime.combine(metrics.readElapsedTime);
            mergeElapsedTime.combine(metrics.mergeElapsedTime);
            writeElapsedTime.combine(metrics.writeElapsedTime);
            totalElapsedTime.combine(metrics.totalElapsedTime);

            readMaxLag.combine(metrics.readMaxLag);
            mergeMaxLag.combine(metrics.mergeMaxLag);
            writeQueue.combine(metrics.writeQueue);
            invalidArchives.combine(metrics.invalidArchives);
        }

        @Override
        public int estimateCount() {
            return registry.estimateCount();
        }

        @Override
        public void append(long tsMillis, Labels commonLabels, MetricConsumer consumer) {
            registry.append(tsMillis, commonLabels, consumer);
            taskMetrics.append(tsMillis, commonLabels, consumer);
        }
    }


    public static class MergeTaskMetrics implements MetricSupplier {
        private final MetricRegistry registry;
        private final Rate started;
        private final Rate completed;
        private final Rate failed;
        private final LazyGaugeInt64 inflight;
        private final Histogram elapsedTimeMillis;
        private final Histogram snapshots;
        private final Histogram frames;

        private final Rate archiveCount;
        private final Rate pointsCount;
        private final Rate pointsCollapsed;
        private final Rate forceDecimCount;

        public MergeTaskMetrics(String kind) {
            registry = new MetricRegistry(Labels.of("kind", kind));

            started = registry.rate("mergeProcess.task.started");
            completed = registry.rate("mergeProcess.task.completed");
            failed = registry.rate("mergeProcess.task.failed");
            inflight = registry.lazyGaugeInt64("mergeProcess.task.inFlight", () -> {
                return started.get() - (completed.get() + failed.get());
            });
            elapsedTimeMillis = registry.histogramRate("mergeProcess.task.elapsedTimeMillis",
                Histograms.exponential(17, 2, 1));
            snapshots = registry.histogramRate("mergeProcess.task.snapshots",
                Histograms.exponential(8, 2));
            frames = registry.histogramRate("mergeProcess.task.frames",
                Histograms.exponential(20, 2));

            archiveCount = registry.rate("mergeProcess.task.archiveCount");
            pointsCount = registry.rate("mergeProcess.task.pointsCount");
            pointsCollapsed = registry.rate("mergeProcess.task.pointsCollapsed");
            forceDecimCount = registry.rate("mergeProcess.task.forceDecimCount");
        }

        public long started() {
            started.inc();
            return System.nanoTime();
        }

        public void completed(long startTimeNanos) {
            long elapsedTime = System.nanoTime() - startTimeNanos;
            completed.inc();
            elapsedTimeMillis.record(TimeUnit.NANOSECONDS.toMillis(elapsedTime));
        }

        public void addCountArchive(long archiveCount) {
            this.snapshots.record(archiveCount);
            this.archiveCount.add(archiveCount);
        }

        public void addCountFrames(long count) {
            this.frames.record(count);
        }

        public void addPointsCount(long pointsCount) {
            this.pointsCount.add(pointsCount);
        }

        public void addCollapsedPoints(long collapsedPoints) {
            pointsCollapsed.add(collapsedPoints);
        }

        public void failed() {
            failed.inc();
        }

        public void addForceDecim() {
            forceDecimCount.inc();
        }

        public void combine(MergeTaskMetrics metrics) {
            started.combine(metrics.started);
            completed.combine(metrics.completed);
            failed.combine(metrics.failed);
            elapsedTimeMillis.combine(metrics.elapsedTimeMillis);
            snapshots.combine(metrics.snapshots);
            frames.combine(metrics.frames);
            archiveCount.combine(metrics.archiveCount);
            pointsCount.combine(metrics.pointsCount);
            pointsCollapsed.combine(metrics.pointsCollapsed);
            forceDecimCount.combine(metrics.forceDecimCount);
        }

        @Override
        public int estimateCount() {
            return registry.estimateCount();
        }

        @Override
        public void append(long tsMillis, Labels commonLabels, MetricConsumer consumer) {
            registry.append(tsMillis, commonLabels, consumer);
        }
    }
}
