package ru.yandex.stockpile.server.shard;

import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Flow;
import java.util.concurrent.SubmissionPublisher;
import java.util.concurrent.atomic.AtomicLong;

import javax.annotation.Nullable;

import ru.yandex.misc.actor.ActorWithFutureRunner;
import ru.yandex.solomon.memory.layout.MemMeasurable;
import ru.yandex.solomon.memory.layout.MemoryCounter;
import ru.yandex.stockpile.kikimrKv.counting.ReadClass;
import ru.yandex.stockpile.memState.MetricIdAndData;
import ru.yandex.stockpile.server.data.index.SnapshotIndex;
import ru.yandex.stockpile.server.shard.MergeProcessMetrics.MergeKindMetrics;
import ru.yandex.stockpile.server.shard.iter.MetricsIterator;
import ru.yandex.stockpile.server.shard.load.AsyncIterator;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static ru.yandex.misc.concurrent.CompletableFutures.isCompletedSuccessfully;

/**
 * @author Vladimir Gordiychuk
 */
public class MergeReader extends SubmissionPublisher<List<MetricIdAndData>> implements MemMeasurable {
    private static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(MergeReader.class);

    private final AtomicLong memoryUsage = new AtomicLong(SELF_SIZE);
    private MergeKindMetrics metrics;
    private AsyncIterator<List<MetricIdAndData>> iterator;
    private ActorWithFutureRunner actor;

    public MergeReader(ShardThread shardThread, SnapshotIndex[] indexes, MergeKindMetrics metrics) {
        super(shardThread.shard.mergeExecutor, 2048);
        this.metrics = metrics;
        this.iterator = new MetricsIterator(shardThread.shard.storage, shardThread, indexes, ReadClass.MERGE_READ_CHUNK);
        this.actor = new ActorWithFutureRunner(this::act, shardThread.shard.mergeExecutor);
    }

    @Override
    public void subscribe(Flow.Subscriber<? super List<MetricIdAndData>> subscriber) {
        super.subscribe(new SubscriberWrapper(subscriber));
    }

    public CompletableFuture<?> act() {
        if (getNumberOfSubscribers() == 0) {
            return completedFuture(null);
        }

        int lag = estimateMaximumLag();
        while (lag < getMaxBufferCapacity()) {
            long startNanos = System.nanoTime();
            var future = iterator.next();

            if (isCompletedSuccessfully(future)) {
                var result = future.getNow(null);
                lag = process(startNanos, result, null);
                continue;
            }

            return future.whenComplete((result, e) -> {
                process(startNanos, result, e);
                actor.schedule();
            });
        }

        return completedFuture(null);
    }

    private int process(long startNanos, List<MetricIdAndData> result, @Nullable Throwable e) {
        if (e != null) {
            closeExceptionally(e);
            return getMaxBufferCapacity();
        }

        if (result == null) {
            close();
            return getMaxBufferCapacity();
        } else {
            memoryUsage.addAndGet(MemoryCounter.listDataSizeWithContent(result));
            metrics.addReadTime(System.nanoTime() - startNanos);
            int lag = submit(result);
            metrics.readMaxLag.record(lag);
            return lag;
        }
    }

    @Override
    public long memorySizeIncludingSelf() {
        return memoryUsage.get();
    }

    private class SubscriberWrapper implements Flow.Subscriber<List<MetricIdAndData>> {
        private final Flow.Subscriber<? super List<MetricIdAndData>> subscriber;

        public SubscriberWrapper(Flow.Subscriber<? super List<MetricIdAndData>> subscriber) {
            this.subscriber = subscriber;
        }

        @Override
        public void onSubscribe(Flow.Subscription subscription) {
            subscriber.onSubscribe(new SubscriptionWrapper(subscription));
        }

        @Override
        public void onNext(List<MetricIdAndData> item) {
            memoryUsage.addAndGet(-MemoryCounter.listDataSizeWithContent(item));
            subscriber.onNext(item);
        }

        @Override
        public void onError(Throwable throwable) {
            subscriber.onError(throwable);
        }

        @Override
        public void onComplete() {
            subscriber.onComplete();
        }
    }

    private class SubscriptionWrapper implements Flow.Subscription {
        private final Flow.Subscription subscription;

        public SubscriptionWrapper(Flow.Subscription subscription) {
            this.subscription = subscription;
        }

        @Override
        public void request(long n) {
            if (n > 0) {
                subscription.request(n);
                actor.schedule();
            }
        }

        @Override
        public void cancel() {
            subscription.cancel();
        }
    }
}
