package ru.yandex.stockpile.server.shard;

import java.time.Clock;
import java.util.Arrays;
import java.util.EnumSet;
import java.util.concurrent.ThreadLocalRandom;
import java.util.stream.Stream;

import ru.yandex.solomon.staffOnly.annotations.LinkedOnRootPage;
import ru.yandex.stockpile.server.SnapshotLevel;
import ru.yandex.stockpile.server.data.index.SnapshotIndex;

/**
 * @author Vladimir Gordiychuk
 */
@LinkedOnRootPage("Merge Strategy")
public class MergeStrategy {
    private final MergeOptions daily;
    private final MergeOptions eternity;
    private final Clock clock;

    public MergeStrategy(MergeOptions daily, MergeOptions eternity, Clock clock) {
        this.daily = daily;
        this.eternity = eternity;
        this.clock = clock;
    }

    public MergeOptions getDaily() {
        return daily;
    }

    public MergeOptions getEternity() {
        return eternity;
    }

    public MergeAdvice chooseMerge(Stream<SnapshotIndexWithStats> targets, MergeKind kind) {
        switch (kind) {
            case DAILY:
                return chooseMerge(targets, kind, daily);
            case ETERNITY:
                return chooseMerge(targets, kind, eternity);
            default:
                throw new UnsupportedOperationException(kind.name());
        }
    }

    private MergeAdvice chooseMerge(Stream<SnapshotIndexWithStats> targets, MergeKind kind, MergeOptions opts) {
        var levels = targetLevelsForMerge(kind, opts);
        var all = targets.toArray(SnapshotIndexWithStats[]::new);
        // sorted by level: eternity, daily, 2h
        var affect = Stream.of(all)
            .filter(target -> levels.contains(target.getIndex().getLevel()))
            .toArray(SnapshotIndexWithStats[]::new);

        if (affect.length == 0) {
            return new MergeAdvice(new SnapshotIndex[0], false, false, 0);
        }

        if (!opts.isEnableNew()) {
            boolean allowDelete = all[0] == affect[0];
            boolean allowDecim = allowDelete;
            var indexes = Stream.of(affect)
                .map(SnapshotIndexWithStats::getIndex)
                .toArray(SnapshotIndex[]::new);

            return new MergeAdvice(indexes, allowDelete, allowDecim, 0);
        }

        long now = clock.millis();
        long forceAfterMillis = opts.getForceMergeAfterMillis();
        if (opts.isEnableForceAfterJitterMillis()) {
            forceAfterMillis += ThreadLocalRandom.current().nextLong(opts.getForceMergeAfterJitterMillis());
        }

        long[] mergedSizes = estimateMergedSize(affect);
        int snapshotCount = 0;
        int pos = 0;
        while (pos < affect.length) {
            var snapshot = affect[pos];

            if (snapshot.getIndex().getLevel() != kind.targetLevel) {
                break;
            }

            long createdAt = snapshot.getIndex().getContent().getTsMillis();
            if (opts.isEnableForceAfterMillis() && (now - createdAt) >= forceAfterMillis) {
                break;
            }

            // avoid merge latest the same chunk without any affect
            if (pos + 1 == affect.length) {
                pos = affect.length;
                break;
            }

            snapshotCount++;
            if (opts.isEnableSnapshotLimit() && snapshotCount >= opts.getSnapshotsLimit()) {
                break;
            }

            long size = diskSize(snapshot);
            if (opts.isEnableMinSnapshotBytesLimit() && opts.getMinSnapshotBytesSize() >= size) {
                break;
            }

            long sizeRight = mergedSizes[pos] - size;
            if (sizeRight >= size) {
                break;
            }

            pos++;
        }

        // absent snapshot to merge
        if (pos >= affect.length) {
            return new MergeAdvice(new SnapshotIndex[0], false, false, 0);
        }

        boolean allowDelete = all[0] == affect[pos];
        boolean allowDecim = pos == 0 && opts.isAllowDecim();
        long splitDelayMillis = pos == 0 ? opts.getSplitDelayMillis() : 0;
        var indexes = Stream.of(affect)
            .skip(pos)
            .map(SnapshotIndexWithStats::getIndex)
            .toArray(SnapshotIndex[]::new);
        return new MergeAdvice(indexes, allowDelete, allowDecim, splitDelayMillis);
    }

    private EnumSet<SnapshotLevel> targetLevelsForMerge(MergeKind kind, MergeOptions opts) {
        if (!opts.isEnableNew()) {
            return kind.readLevels();
        }

        if (opts.isUsePrevLevels()) {
            return EnumSet.copyOf(Arrays.asList(kind.targetLevel.currentAndAboveLevels()));
        } else {
            return EnumSet.of(kind.targetLevel);
        }
    }

    private long[] estimateMergedSize(SnapshotIndexWithStats[] targets) {
        var result = new long[targets.length];

        long sum = 0;
        for (int index = targets.length - 1; index >= 0; index--) {
            var target = targets[index];
            long weight = diskSize(target);
            sum += weight;
            result[index] = sum;
        }
        return result;
    }

    private static long diskSize(SnapshotIndexWithStats target) {
        var diskSize = target.diskSize();
        return diskSize.chunk().size() + diskSize.index().size() + diskSize.command().size();
    }
}
