package ru.yandex.stockpile.server.shard;

import java.util.concurrent.atomic.AtomicReferenceFieldUpdater;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.misc.lang.Verify;
import ru.yandex.solomon.memory.layout.MemMeasurable;
import ru.yandex.solomon.staffOnly.manager.special.InstantMillis;
import ru.yandex.stockpile.server.shard.actor.ActorRunnable;
import ru.yandex.stockpile.server.shard.actor.InActor;

/**
 * @author Stepan Koltsov
 */
abstract class ShardProcess extends ShardThread implements MemMeasurable {

    private static final Logger logger = LoggerFactory.getLogger(ShardProcess.class);


    protected String formatLogPrefix(String message) {
        return processType + " for shard " + shard.shardId + (message.isEmpty() ? "" : " " + message);
    }

    protected void log(String message) {
        logger.info(formatLogPrefix(message));
    }

    protected void logStart(String message) {
        logger.info(formatLogPrefix(message) + " started...");
    }

    protected void logCompletion(String message, long d) {
        logger.info(formatLogPrefix(message) + " completed in " + d + " ms");
    }


    @InstantMillis
    protected final long startedMillis;
    private final ProcessType processType;
    private final String message;


    protected ShardProcess(StockpileShard shard, ProcessType processType, String message) {
        super(shard);
        this.processType = processType;
        this.message = message;

        shard.globals.stockpileShardAggregatedStats.processStarted(processType);

        startedMillis = System.currentTimeMillis();

        logStart(message);
    }

    abstract void start(InActor a);

    protected void completedSuccessfullyWriteStats() {
        long d = System.currentTimeMillis() - startedMillis;
        logCompletion(message, d);
    }


    private volatile ActorRunnable runInActor;

    private static final AtomicReferenceFieldUpdater<ShardProcess, ActorRunnable> runInActorField =
        AtomicReferenceFieldUpdater.newUpdater(ShardProcess.class, ActorRunnable.class, "runInActor");

    protected void processRunInActor(ActorRunnable next) {
        ActorRunnable prev = runInActorField.getAndSet(this, next);
        Verify.V.isNull(prev);
        shard.actorRunner.schedule();
    }

    void act(InActor a) {
        for (;;) {
            ActorRunnable runInActor = runInActorField.getAndSet(this, null);
            if (runInActor == null) {
                return;
            }
            runInActor.run(a);
        }
    }
}
