package ru.yandex.stockpile.server.shard;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.kikimr.client.kv.KikimrKvGenerationChangedRuntimeException;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.util.ExceptionUtils;

import static ru.yandex.solomon.util.time.DurationUtils.backoff;
import static ru.yandex.solomon.util.time.DurationUtils.randomize;
import static ru.yandex.stockpile.server.shard.ExceptionHandler.isGenerationChanged;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public abstract class ShardThread {

    private final int SLEEP_MIN_MILLIS = 5_000;
    private final int SLEEP_MAX_MILLIS = 15_000;

    protected final StockpileShard shard;

    public ShardThread(StockpileShard shard) {
        this.shard = shard;
    }

    protected abstract void stoppedReleaseResources();

    protected void safeCallStopped() {
        try {
            stoppedReleaseResources();
        } catch (Throwable x) {
            ExceptionUtils.uncaughtException(x);
        }
    }

    Throwable lastError;

    private void error(Throwable error) {
        shard.error(error);
        lastError = error;
    }

    private <A> void    loopUntilSuccessFutureImpl(
        String desc,
        Supplier<CompletableFuture<A>> op,
        CompletableFuture<A> resultFuture,
        int attempt)
    {
        if (shard.stop) {
            safeCallStopped();
            resultFuture.completeExceptionally(new KikimrKvGenerationChangedRuntimeException(shard.kvTabletId));
            return;
        }

        CompletableFutures.safeCall(op::get).whenComplete((r, x) -> {
            try {
                if (isGenerationChanged(x)) {
                    generationChanged();
                    resultFuture.completeExceptionally(x);
                    return;
                }

                if (x != null) {
                    error(new RuntimeException(desc + " failed, attempt " + attempt, CompletableFutures.unwrapCompletionException(x)) {
                        @Override
                        public synchronized Throwable fillInStackTrace() {
                            return this;
                        }
                    });

                    if (attempt == 0) {
                        loopUntilSuccessFutureImpl(desc, op, resultFuture, attempt + 1);
                    } else {
                        var delay = randomize(backoff(SLEEP_MIN_MILLIS, SLEEP_MAX_MILLIS, attempt));
                        schedule(() -> {
                            shard.commonExecutor.execute(() -> {
                                loopUntilSuccessFutureImpl(desc, op, resultFuture, attempt + 1);
                            });
                        }, delay);
                    }
                    return;
                }

                resultFuture.complete(r);
            } catch (Throwable x2) {
                ExceptionUtils.uncaughtException(x2);
            }
        });
    }

    private void schedule(Runnable runnable, long delayMillis) {
        shard.globals.scheduledExecutorService.schedule(runnable, delayMillis, TimeUnit.MILLISECONDS);
    }

    public <A> CompletableFuture<A> loopUntilSuccessFuture(String desc, Supplier<CompletableFuture<A>> op) {
        CompletableFuture<A> r = new CompletableFuture<>();
        loopUntilSuccessFutureImpl(desc, op, r, 0);
        return r;
    }

    protected void generationChanged() {
        shard.generationChanged = true;
        shard.stop();
        safeCallStopped();
        shard.actorRunner.schedule();
    }
}
