package ru.yandex.stockpile.server.shard;

import java.time.Duration;
import java.util.EnumMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicIntegerFieldUpdater;
import java.util.stream.Stream;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Maps;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.misc.actor.ActorRunner;
import ru.yandex.solomon.staffOnly.annotations.LinkedOnRootPage;
import ru.yandex.solomon.staffOnly.annotations.ManagerMethod;
import ru.yandex.solomon.staffOnly.annotations.ManagerMethodArgument;
import ru.yandex.solomon.util.ExceptionUtils;
import ru.yandex.solomon.util.concurrent.ThreadUtils;
import ru.yandex.stockpile.server.SnapshotLevel;

/**
 * @author Stepan Koltsov
 */
@Component
@LinkedOnRootPage("Snapshot & Merge Scheduler")
@ParametersAreNonnullByDefault
public class SnapshotAndMergeScheduler implements AutoCloseable {
    private final EnumMap<SnapshotLevel, SnapshotStats> levelStats;
    private final StockpileLocalShards shards;
    private final MergeStrategy mergeStrategy;
    private final ActorRunner actor;
    private final ScheduledFuture<?> pereodicallyAct;
    private volatile boolean closed;

    @Autowired
    public SnapshotAndMergeScheduler(
        StockpileLocalShards shards,
        @StockpileScheduledExecutor ScheduledExecutorService timer,
        @StockpileExecutor ExecutorService executor,
        MergeStrategy mergeStrategy)
    {
        EnumMap<SnapshotLevel, SnapshotStats> levelStats = Maps.newEnumMap(SnapshotLevel.class);
        for (SnapshotLevel level : SnapshotLevel.values()) {
            levelStats.put(level, new SnapshotStats());
        }
        this.levelStats = levelStats;
        this.shards = shards;
        this.mergeStrategy = mergeStrategy;
        this.actor = new ActorRunner(this::act, executor);
        this.pereodicallyAct = timer.scheduleAtFixedRate(actor::schedule, 0, 5, TimeUnit.SECONDS);
    }

    private Stream<StockpileShard> localShardsReadyAndWithoutProcesses() {
        return shards.stream()
                .filter(s -> !s.isMergeOrSnapshotInProgress())
                .filter(s -> s.getLoadState() == StockpileShard.LoadState.DONE);
    }

    private void handleAsyncProcessResult(
        StockpileShard shard, @Nullable Object r, @Nullable Throwable x)
    {
        if (x != null) {
            if (!shard.stop) {
                ExceptionUtils.uncaughtException(new Exception(x));
            }
        }
    }

    private void act() {
        if (closed) {
            return;
        }

        scheduleTwoHourSnapshotIfNecessary();
        for (MergeKind mergeKind : MergeKind.values()) {
            scheduleMergeIfNecessary(mergeKind);
        }
    }

    private void scheduleTwoHourSnapshotIfNecessary() {
        var stats = levelStats.get(SnapshotLevel.TWO_HOURS);
        if (stats.inFlight >= stats.maxInFlight) {
            return;
        }

        if (isExistsLoadingShards()) {
            return;
        }

        long now = System.currentTimeMillis();

        StockpileShard oldestSnapshotShard = null;
        long oldestSnapshotTs = Long.MAX_VALUE;

        var it = localShardsReadyAndWithoutProcesses().iterator();
        while (it.hasNext()) {
            var shard = it.next();

            long snapshotTimeMillis = snapshotTs(shard, SnapshotLevel.TWO_HOURS);
            if (now - snapshotTimeMillis < shard.snapshotPeriodMillis) {
                continue;
            }

            if (snapshotTimeMillis <= oldestSnapshotTs) {
                oldestSnapshotShard = shard;
                oldestSnapshotTs = snapshotTimeMillis;
            }
        }

        if (oldestSnapshotShard == null) {
            return;
        }

        final var shard = oldestSnapshotShard;
        stats.incInFlight();
        oldestSnapshotShard.forceSnapshot().whenComplete((r, x) -> {
            stats.decInFlight();
            handleAsyncProcessResult(shard, r, x);
            actor.schedule();
        });
    }

    static long randomSnapshotPeriod() {
        return ThreadUtils.currentThreadLocalRandom().nextLong(
            Duration.ofMinutes(90).toMillis(),
            Duration.ofMinutes(150).toMillis());
    }

    private boolean isExistsLoadingShards() {
        for (var shard : shards) {
            if (shard.getLoadState() != StockpileShard.LoadState.DONE) {
                return true;
            }
        }

        return false;
    }

    private void scheduleMergeIfNecessary(MergeKind mergeKind) {
        var stats = levelStats.get(mergeKind.targetLevel);
        if (stats.inFlight >= stats.maxInFlight) {
            return;
        }

        if (isExistsLoadingShards()) {
            return;
        }

        StockpileShard oldestMergeShard = chooseShardToMerge(mergeKind);
        if (oldestMergeShard == null) {
            return;
        }

        stats.incInFlight();
        oldestMergeShard.forceMerge(mergeKind).whenComplete((r, x) -> {
            stats.decInFlight();
            handleAsyncProcessResult(oldestMergeShard, r, x);
            actor.schedule();
        });
    }

    @Nullable
    private StockpileShard chooseShardToMerge(MergeKind mergeKind) {
        switch (mergeKind) {
            case DAILY:
                return chooseShardForDailyMerge();
            case ETERNITY:
                return chooseShardForEternityMerge();
            default:
                throw new AssertionError();
        }
    }

    @Nullable
    private StockpileShard chooseShardForEternityMerge() {
        var now = System.currentTimeMillis();
        var candidate = localShardsReadyAndWithoutProcesses()
                .map(MergeCandidate::eternity)
                .filter(s -> {
                    // at least once a day trigger eternity merge, event if not enough snapshots to merge
                    if (now - s.latestSnapshotTs >= TimeUnit.DAYS.toMillis(1)) {
                        return true;
                    }

                    // avoid make merge if nothing to merge
                    return s.snapshotCount > mergeStrategy.getEternity().getSnapshotsLimit();
                })
                .max(MergeCandidate::compareTo)
                .orElse(null);

        if (candidate == null) {
            return null;
        }

        return candidate.shard;
    }

    @Override
    public void close() {
        closed = true;
        pereodicallyAct.cancel(false);
    }

    @ManagerMethod
    public void setMaxInFlight(
        @ManagerMethodArgument(name = "level") SnapshotLevel level,
        @ManagerMethodArgument(name = "maxInFlight") int maxInFlight)
    {
        levelStats.get(level).maxInFlight = maxInFlight;
    }

    @Nullable
    private StockpileShard chooseShardForDailyMerge() {
        var now = System.currentTimeMillis();
        var stats = localShardsReadyAndWithoutProcesses()
                .map(MergeCandidate::daily)
                .filter(s -> {
                    // at least once a day trigger daily merge, event if not enough snapshots to merge
                    if (now - s.latestSnapshotTs >= TimeUnit.DAYS.toMillis(1)) {
                        return true;
                    }

                    return s.snapshotCount > 1;
                })
                .max(MergeCandidate::compareTo)
                .orElse(null);

        if (stats == null) {
            return null;
        }

        return stats.shard;
    }

    private static long snapshotTs(StockpileShard shard, SnapshotLevel level) {
        return shard.latestSnapshotTime(level)
                .getTsOrNever()
                .orElse(shard.stockpileShardCreatedInstantMillis);
    }

    static record MergeCandidate(StockpileShard shard, int snapshotCount, long latestSnapshotTs) implements Comparable<MergeCandidate> {
        public static MergeCandidate daily(StockpileShard shard) {
            int snapshotCount = shard.snapshotCount(SnapshotLevel.TWO_HOURS).orElse(0);
            long dailySnapshotTs = snapshotTs(shard, SnapshotLevel.DAILY);

            return new MergeCandidate(shard, snapshotCount, dailySnapshotTs);
        }

        public static MergeCandidate eternity(StockpileShard shard) {
            int snapshotCount = shard.snapshotCount(SnapshotLevel.ETERNITY).orElse(0);
            long eternitySnapshotTs = snapshotTs(shard, SnapshotLevel.ETERNITY);

            return new MergeCandidate(shard, snapshotCount, eternitySnapshotTs);
        }

        @Override
        public int compareTo(@Nonnull MergeCandidate o) {
            // reversed compare, less snapshot ts increase priority
            int compare = Long.compare(
                    TimeUnit.MILLISECONDS.toDays(o.latestSnapshotTs),
                    TimeUnit.MILLISECONDS.toDays(latestSnapshotTs));

            if (compare != 0) {
                return compare;
            }

            // more snapshot count increase priority
            compare = Integer.compare(snapshotCount, o.snapshotCount);
            if (compare != 0) {
                return compare;
            }

            // reversed compare, less snapshot ts increase priority
            return Long.compare(o.latestSnapshotTs, latestSnapshotTs);
        }
    }

    private static class SnapshotStats {
        private static final AtomicIntegerFieldUpdater<SnapshotStats> inFlightField =
                AtomicIntegerFieldUpdater.newUpdater(SnapshotStats.class, "inFlight");

        volatile int inFlight;
        volatile int maxInFlight = 1;

        void incInFlight() {
            inFlightField.incrementAndGet(this);
        }

        void decInFlight() {
            inFlightField.decrementAndGet(this);
        }

        @Override
        public String toString() {
            return "SnapshotStats{" +
                    "inFlight=" + inFlight +
                    ", maxInFlight=" + maxInFlight +
                    '}';
        }
    }
}
