package ru.yandex.stockpile.server.shard;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import com.google.protobuf.ByteString;
import com.google.protobuf.UnsafeByteOperations;

import ru.yandex.kikimr.client.kv.KvReadRangeResult;
import ru.yandex.kikimr.client.kv.StringMicroUtils;
import ru.yandex.kikimr.util.NameRange;
import ru.yandex.solomon.config.protobuf.stockpile.EInvalidArchiveStrategy;
import ru.yandex.stockpile.kikimrKv.counting.ReadClass;
import ru.yandex.stockpile.server.data.chunk.SnapshotAddress;
import ru.yandex.stockpile.server.data.command.LongFileTypeSnapshotCommand;
import ru.yandex.stockpile.server.data.command.SnapshotCommand;
import ru.yandex.stockpile.server.data.names.StockpileKvNames;
import ru.yandex.stockpile.server.data.names.file.CommandFile;
import ru.yandex.stockpile.server.shard.load.Async;
import ru.yandex.stockpile.server.shard.load.KvReadRangeIterator;

/**
 * @author Vladimir Gordiychuk
 */
public class SnapshotCommandLoaderImpl implements SnapshotCommandLoader {
    private final ShardThread shardThread;

    public SnapshotCommandLoaderImpl(ShardThread shardThread) {
        this.shardThread = shardThread;
    }

    @Override
    public CompletableFuture<Optional<SnapshotCommand>> readSnapshot(SnapshotAddress address) {
        var prefix = StockpileKvNames.commandPrefixCurrent(address);
        var range = StringMicroUtils.asciiPrefixToRange(prefix);
        var it = new CommandReadRangeIterator(range);
        var chunks = new ArrayList<Chunk>();
        return Async.forEach(it, next -> {
            var file = CommandFile.pfCurrent.parseOrThrow(next.getName());
            chunks.add(new Chunk(file, UnsafeByteOperations.unsafeWrap(next.getValue())));
        }).thenApply(ignore -> {
            if (chunks.isEmpty()) {
                return Optional.empty();
            }

            try {
                ensureValid(chunks, address);
                var data = chunks.stream().map(Chunk::data).toArray(ByteString[]::new);
                var content = LongFileTypeSnapshotCommand.I.serializer().deserializeParts(data);
                return Optional.of(new SnapshotCommand(address.level(), address.txn(), content));
            } catch (Throwable e) {
                var file = CommandFile.currentSnapshotDotTxDotPf.format(address);
                var shard = shardThread.shard;
                var error = new RuntimeException("failed to parser file: " + file, e);
                shard.error(error);

                if (shard.globals.invalidArchiveStrategy.strategy == EInvalidArchiveStrategy.DROP) {
                    return Optional.empty();
                }

                throw error;
            }
        });
    }

    private void ensureValid(List<Chunk> chunks, SnapshotAddress address) {
        for (int index = 0; index < chunks.size(); index++) {
            var chunk = chunks.get(index);
            if (!isValid(chunk.file, address, index, index + 1 == chunks.size())) {
                throw new IllegalStateException("Snapshot corrupted: " + chunks.stream()
                        .map(c -> CommandFile.pfCurrent.format(c.file))
                        .collect(Collectors.joining()));
            }
        }
    }

    private boolean isValid(CommandFile file, SnapshotAddress address, int part, boolean last) {
        if (file.getLevel() != address.level()) {
            return false;
        }

        if (file.txn() != address.txn()) {
            return false;
        }

        if (file.getPartNo() != part) {
            return false;
        }

        return file.isLast() == last;
    }

    /**
     * Read iterator with retries and counting total files size.
     */
    private final class CommandReadRangeIterator extends KvReadRangeIterator {
        CommandReadRangeIterator(NameRange nameRange) {
            super(nameRange);
        }

        @Override
        protected CompletableFuture<KvReadRangeResult> readNext(NameRange nameRange) {
            return shardThread.loopUntilSuccessFuture("readCommand", () -> shardThread.shard.storage.readRange(ReadClass.OTHER, nameRange));
        }
    }

    private record Chunk(CommandFile file, ByteString data) {
    }
}
