package ru.yandex.stockpile.server.shard;

import java.time.Clock;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ScheduledExecutorService;

import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

import ru.yandex.solomon.config.protobuf.stockpile.TStockpileConfig;
import ru.yandex.solomon.config.protobuf.stockpile.TStockpileThreadPoolsConfig;
import ru.yandex.solomon.config.thread.ThreadPoolProvider;
import ru.yandex.solomon.util.time.TimeProvider;
import ru.yandex.solomon.util.time.TimeProviderImpl;
import ru.yandex.stockpile.api.read.StockpileReadApi;
import ru.yandex.stockpile.server.StockpileKikimrClientContext;
import ru.yandex.stockpile.server.shard.cache.StockpileCacheManager;
import ru.yandex.stockpile.server.www.StockpileLocalShardsSnapshotWww;
import ru.yandex.stockpile.server.www.StockpileLocalShardsWww;
import ru.yandex.stockpile.server.www.StockpileResourceUsageController;

/**
 * @author Stepan Koltsov
 */
@Configuration
@Import({
    StockpileLocalShardsWww.class,
    StockpileLocalShardsSnapshotWww.class,
    StockpileShardLoader.class,
    StockpileShardsHealthChecker.class,
    SnapshotAndMergeScheduler.class,
    StockpileLocalShardsMemoryManager.class,
    StockpileCacheManager.class,
    StockpileKikimrClientContext.class,
    StockpileReadApi.class,
    StockpileResourceUsageController.class,
})
public class StockpileContext {

    private final ThreadPoolProvider threadPoolProvider;
    private final TStockpileThreadPoolsConfig config;

    public StockpileContext(
        ThreadPoolProvider threadPoolProvider,
        TStockpileThreadPoolsConfig config)
    {
        this.threadPoolProvider = threadPoolProvider;
        this.config = config;
    }

    @StockpileExecutor
    @Bean
    public ExecutorService stockpileExecutor() {
        return threadPoolProvider.getExecutorService(config.getGeneralThreadPool(), "StockpileThreadPoolsConfig.GeneralThreadPool");
    }

    @StockpileScheduledExecutor
    @Bean
    public ScheduledExecutorService stockpileScheduler() {
        return threadPoolProvider.getSchedulerExecutorService();
    }

    @StockpileMergeExecutor
    @Bean
    public ExecutorService mergeExecutor() {
        return threadPoolProvider.getExecutorService(
            config.getMergeThreadPool(),
            "StockpileThreadPoolsConfig.MergeThreadPool");
    }

    @StockpileBalancerExecutor
    @Bean
    public ExecutorService balancerExecutor() {
        return threadPoolProvider.getExecutorService(
            config.getBalancerThreadPool(),
            "StockpileThreadPoolsConfig.BalancerThreadPool");
    }

    @Bean
    @StockpileReadExecutor
    public ExecutorService stockpileReadExecutor() {
        return threadPoolProvider.getExecutorService(
            config.getStockpileReadThreadPool(),
            "StockpileThreadPoolsConfig.StockpileReadThreadPool");
    }

    @Bean
    public Clock clock() {
        return Clock.systemUTC();
    }

    @Bean
    public TimeProvider timeProvider() {
        return new TimeProviderImpl();
    }

    @Bean
    public InvalidArchiveStrategy invalidArchiveStrategy(TStockpileConfig config) {
        return new InvalidArchiveStrategy(config.getInvalidArchiveStrategy());
    }
}
