package ru.yandex.stockpile.server.shard;

import java.util.Arrays;
import java.util.Iterator;
import java.util.NoSuchElementException;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import io.grpc.Status;
import org.springframework.context.event.ContextClosedEvent;
import org.springframework.context.event.EventListener;
import org.springframework.core.annotation.Order;
import org.springframework.stereotype.Component;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.memory.layout.MemInfoProvider;
import ru.yandex.solomon.memory.layout.MemoryBySubsystem;
import ru.yandex.solomon.staffOnly.manager.find.annotation.NamedObjectFinderAnnotation;
import ru.yandex.stockpile.server.data.names.FileKind;

import static java.util.stream.Collectors.toList;

/**
 * @author Vladimir Gordiychuk
 */
@Component
@ParametersAreNonnullByDefault
public class StockpileLocalShards implements MemInfoProvider, Iterable<StockpileShard> {
    // COW
    private volatile StockpileShard[] shards = new StockpileShard[0];
    private volatile boolean closed;
    private volatile int size;

    @Nullable
    @NamedObjectFinderAnnotation
    public StockpileShard getShardById(int shardId) {
        StockpileShard shard = get(shardId);
        if (shard != null && shard.isStop()) {
            remove(shard);
            return null;
        }
        return shard;
    }

    @Nullable
    @NamedObjectFinderAnnotation
    public StockpileShardInspector getInspector(int shardId) {
        StockpileShard shard = get(shardId);
        if (shard != null && !shard.isStop()) {
            return new StockpileShardInspector(shard);
        }

        return null;
    }

    public void ensureCapacity(int totalShards) {
        if (shards.length >= totalShards) {
            return;
        }

        synchronized (this) {
            if (shards.length >= totalShards) {
                return;
            }

            shards = Arrays.copyOf(shards, Math.max(shards.length, totalShards));
        }
    }

    @Nullable
    private StockpileShard get(int shardId) {
        ensureShardIdValid(shardId);
        var copy = shards;
        if (copy.length < shardId) {
            return null;
        }

        return copy[idx(shardId)];
    }

    private void ensureShardIdValid(int shardId) {
        if (shardId <= 0) {
            throw new IllegalStateException("Invalid shard id: " + shardId);
        }
    }

    private boolean compareAndSet(int shardId, @Nullable StockpileShard prev, @Nullable StockpileShard next) {
        ensureShardIdValid(shardId);
        synchronized (this) {
            if (!Objects.equals(get(shardId), prev)) {
                return false;
            }

            var copy = Arrays.copyOf(shards, Math.max(shards.length, shardId));
            copy[idx(shardId)] = next;
            size += next == null ? -1 : 1;
            shards = copy;
            return true;
        }
    }

    public boolean addShard(StockpileShard shard) {
        if (closed) {
            throw Status.ABORTED.withDescription("Process graceful shutdown").asRuntimeException();
        }

        return compareAndSet(shard.shardId, null, shard);
    }

    public boolean remove(StockpileShard shard) {
        if (closed) {
            throw Status.ABORTED.withDescription("Process graceful shutdown").asRuntimeException();
        }

        return compareAndSet(shard.shardId, shard, null);
    }

    public Stream<StockpileShard> stream() {
        var copy = shards;
        return Stream.of(copy)
                .filter(Objects::nonNull)
                .filter(shard -> {
                    if (!shard.isStop()) {
                        return true;
                    }
                    removeIfExists(shard);
                    return false;
                });
    }

    public int size() {
        return size;
    }

    public int totalShardsCount() {
        return shards.length;
    }

    @Override
    public MemoryBySubsystem memoryBySystem() {
        MemoryBySubsystem r = new MemoryBySubsystem();
        stream().forEach(shard -> shard.addMemoryBySubsystem(r));
        return r;
    }

    @Override
    @Nonnull
    public Iterator<StockpileShard> iterator() {
        return new Iter(shards);
    }

    @Order(1)
    @EventListener(ContextClosedEvent.class)
    public void gracefulShutdownSync() {
        // https://jira.spring.io/browse/SPR-17298
        gracefulShutdown().join();
    }

    public CompletableFuture<Void> gracefulShutdown() {
        closed = true;
        return stream()
            .map(shard -> {
                // don't accept new requests
                compareAndSet(shard.shardId, shard, null);
                if (shard.diskStats().get(FileKind.LOG).size() >= (100 << 20)) {
                    return shard.forceSnapshot().whenComplete((ignore, e) -> shard.stop());
                }

                shard.stop();
                return CompletableFuture.completedFuture(null);
            })
            .collect(Collectors.collectingAndThen(toList(), CompletableFutures::allOfVoid));
    }

    private static int idx(int shardId) {
        return shardId - 1;
    }

    private void removeIfExists(StockpileShard shard) {
        var prev = getShardById(shard.shardId);
        if (prev == shard) {
            compareAndSet(shard.shardId, shard, null);
        }
    }

    private class Iter implements Iterator<StockpileShard> {
        final StockpileShard[] shards;
        // index of next element to return
        int cursor;

        public Iter(StockpileShard[] shards) {
            this.shards = shards;
        }

        @Override
        public boolean hasNext() {
            while (cursor < shards.length) {
                var shard = shards[cursor];
                if (shard == null) {
                    cursor++;
                    continue;
                }

                if (shard.isStop()) {
                    removeIfExists(shard);
                    cursor++;
                    continue;
                }

                return true;
            }
            return false;
        }

        @Override
        public StockpileShard next() {
            if (cursor >= shards.length) {
                throw new NoSuchElementException();
            }
            return shards[cursor++];
        }
    }
}
