package ru.yandex.stockpile.server.shard;

import java.util.concurrent.ExecutorService;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import org.springframework.stereotype.Component;

import ru.yandex.misc.actor.ActorRunner;

/**
 * @author Vladimir Gordiychuk
 */
@Component
public class StockpileLocalShardsMemoryManager implements UpdateShardMemoryLimits, AutoCloseable {
    private static final double CHANGE_PERCENT = 0.1;
    private final StockpileMemory memory;
    private final StockpileLocalShards shards;
    private final ActorRunner actor;
    private final ScheduledFuture<?> future;

    public StockpileLocalShardsMemoryManager(
        StockpileMemory memory,
        StockpileLocalShards shards,
        @StockpileExecutor ExecutorService executor,
        @StockpileScheduledExecutor ScheduledExecutorService time)
    {
        this.shards = shards;
        this.memory = memory;
        this.actor = new ActorRunner(this::act, executor);
        this.future = time.scheduleAtFixedRate(actor::schedule, 0, 5, TimeUnit.SECONDS);
    }

    @Override
    public void updateShardMemoryLimits() {
        actor.schedule();
    }

    private long shardWeight(StockpileShard shard) {
        return shard.recordCount().orElse(0);
    }

    private void act() {
        var localShards = shards.stream().collect(Collectors.toList());
        double sumWeight = localShards.stream().mapToLong(this::shardWeight).sum();

        long minMemoryLimit = memory.getMinShardMemoryLimit();
        for (var shard : localShards) {
            long memoryForShard = 0;
            long weight = shardWeight(shard);
            if (sumWeight > 0) {
                memoryForShard = (long) (1.0 * StockpileMemory.getTotalShardsMemoryLimit() * weight / sumWeight);
            }

            if (memoryForShard <= shard.minRequiredMemory() + minMemoryLimit) {
                memoryForShard = shard.minRequiredMemory() + minMemoryLimit;
                shard.setMemoryLimit(memoryForShard);
                continue;
            }

            long currentLimit = shard.memoryLimit;
            if (currentLimit > memoryForShard) {
                memoryForShard = Math.max(memoryForShard, currentLimit - Math.round(currentLimit * CHANGE_PERCENT));
            } else if (currentLimit < memoryForShard) {
                memoryForShard = Math.min(memoryForShard, currentLimit + Math.round(currentLimit * CHANGE_PERCENT));
            }
            shard.setMemoryLimit(memoryForShard);
        }
    }

    @Override
    public void close() {
        future.cancel(false);
    }
}
