package ru.yandex.stockpile.server.shard;

import java.util.Optional;

import it.unimi.dsi.fastutil.HashCommon;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.dataSize.DataSizeUnit;
import ru.yandex.solomon.config.DataSizeConverter;
import ru.yandex.solomon.config.protobuf.DataUnit;
import ru.yandex.solomon.config.protobuf.stockpile.TMemoryConfig;
import ru.yandex.solomon.staffOnly.annotations.LinkedOnRootPage;
import ru.yandex.solomon.staffOnly.annotations.ManagerMethod;
import ru.yandex.solomon.staffOnly.manager.table.TableColumn;

/**
 * @author Vladimir Gordiychuk
 */
@Component
@LinkedOnRootPage("Memory manager")
public class StockpileMemory {
    private static final long TOTAL_MEMORY = Runtime.getRuntime().maxMemory();
    private static volatile long TOTAL_CACHE_MEMORY_LIMIT;
    private static volatile long TOTAL_SHARDS_MEMORY_LIMIT;
    private static volatile long MIN_SHARD_MEMORY_LIMIT;

    static {
        updateLimits(TMemoryConfig.getDefaultInstance());
    }

    private final StockpileLocalShards localShards;

    @Autowired
    public StockpileMemory(Optional<TMemoryConfig> config, StockpileLocalShards localShards) {
        config.ifPresent(StockpileMemory::updateLimits);
        this.localShards = localShards;
    }

    private static void updateLimits(TMemoryConfig config) {
        long cacheLimit = bytesCount(config.getTotalCacheMemoryLimit());
        if (cacheLimit > 0) {
            TOTAL_CACHE_MEMORY_LIMIT = cacheLimit;
        } else {
            TOTAL_CACHE_MEMORY_LIMIT = TOTAL_MEMORY / 3;
        }

        long shardLimit = bytesCount(config.getTotalShardMemoryLimit());
        if (shardLimit > 0) {
            TOTAL_SHARDS_MEMORY_LIMIT = shardLimit;
        } else {
            TOTAL_SHARDS_MEMORY_LIMIT = Math.round((TOTAL_MEMORY - TOTAL_CACHE_MEMORY_LIMIT) * 0.7);
        }

        MIN_SHARD_MEMORY_LIMIT = bytesCount(config.getMinShardMemoryLimit());
    }

    private static long bytesCount(ru.yandex.solomon.config.protobuf.DataSize dataSize) {
        return DataSizeConverter.convert(dataSize.getValue(), dataSize.getUnit(), DataUnit.BYTES);
    }

    static {
        if ((TOTAL_CACHE_MEMORY_LIMIT > 1000L << 30) || (TOTAL_CACHE_MEMORY_LIMIT < 0)) {
            throw new RuntimeException("sanity check");
        }
    }

    public long getMinShardMemoryLimit() {
        if (MIN_SHARD_MEMORY_LIMIT != 0) {
            return MIN_SHARD_MEMORY_LIMIT;
        }

        int capacity = Math.min(HashCommon.nextPowerOfTwo(localShards.size()), 32) * 32;
        return Math.min(TOTAL_SHARDS_MEMORY_LIMIT / capacity, 10L << 20); // or 10 MiB
    }

    public static long getHostMemoryUsage() {
        return Runtime.getRuntime().totalMemory() - Runtime.getRuntime().freeMemory();
    }

    @TableColumn
    public String totalMemory() {
        return DataSize.shortString(TOTAL_MEMORY);
    }

    @TableColumn
    public String cacheMemory() {
        return DataSize.shortString(TOTAL_CACHE_MEMORY_LIMIT);
    }

    @TableColumn
    public String shardMemory() {
        return DataSize.shortString(TOTAL_SHARDS_MEMORY_LIMIT);
    }

    @TableColumn
    public String minShardMemory() {
        return DataSize.shortString(getMinShardMemoryLimit());
    }

    public static long getTotalCacheMemoryLimit() {
        return TOTAL_CACHE_MEMORY_LIMIT;
    }

    public static long getTotalShardsMemoryLimit() {
        return TOTAL_SHARDS_MEMORY_LIMIT;
    }

    @ManagerMethod
    public void setTotalCacheMemoryLimit(long value, DataSizeUnit unit) {
        TOTAL_CACHE_MEMORY_LIMIT = unit.dataSize(value).toBytes();
    }

    @ManagerMethod
    public void setTotalShardMemoryLimit(long value, DataSizeUnit unit) {
        TOTAL_SHARDS_MEMORY_LIMIT = unit.dataSize(value).toBytes();
    }

    @ManagerMethod
    public void setMinShardMemoryLimit(long value, DataSizeUnit unit) {
        MIN_SHARD_MEMORY_LIMIT = unit.dataSize(value).toBytes();
    }
}
