package ru.yandex.stockpile.server.shard;

import java.time.Instant;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.common.RequestProducer;
import ru.yandex.solomon.memory.layout.MemoryCounter;
import ru.yandex.solomon.staffOnly.manager.special.InstantMillis;
import ru.yandex.stockpile.client.shard.StockpileMetricId;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class StockpileMetricReadRequest implements StockpileRequest {
    private static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(StockpileMetricReadRequest.class);

    private final int shardId;
    private final long localId;
    @InstantMillis
    private final long fromMillis;
    // 0 means +inf
    @InstantMillis
    private final long toMillis;
    @InstantMillis
    private final long deadline;
    private final CompletableFuture<StockpileMetricReadResponse> future;
    private final long createdAtNanos;
    private final RequestProducer producer;

    public StockpileMetricReadRequest(Builder builder) {
        this.shardId = builder.shardId;
        this.localId = builder.localId;
        this.fromMillis = builder.fromMillis;
        this.toMillis = builder.toMillis;
        this.deadline = builder.deadline != 0
                ? builder.deadline
                : System.currentTimeMillis() + TimeUnit.SECONDS.toMillis(30);
        this.producer = builder.producer != null
            ? builder.producer
            : RequestProducer.REQUEST_PRODUCER_UNSPECIFIED;
        this.future = new CompletableFuture<>();
        this.createdAtNanos = System.nanoTime();
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public int getShardId() {
        return shardId;
    }

    public long getLocalId() {
        return localId;
    }

    public long getFromMillis() {
        return fromMillis;
    }

    public long getToMillis() {
        return toMillis;
    }

    public long getDeadline() {
        return deadline;
    }

    public RequestProducer getProducer() {
        return producer;
    }

    public CompletableFuture<StockpileMetricReadResponse> getFuture() {
        return future;
    }

    @Override
    public long memorySizeIncludingSelf() {
        return SELF_SIZE + MemoryCounter.CompletableFuture_SELF_SIZE;
    }

    @Override
    public long getCreatedAtNanos() {
        return createdAtNanos;
    }

    @Override
    public String toString() {
        return StockpileMetricId.toString(shardId, localId) +
                ", interval [" + Instant.ofEpochMilli(fromMillis) + ", " + Instant.ofEpochMilli(toMillis) + "]" +
                ", deadline " + Instant.ofEpochMilli(deadline) +
                ", producer " + producer;
    }

    public static class Builder {
        private int shardId;
        private long localId;
        private long fromMillis;
        // 0 means +inf
        private long toMillis;
        private long deadline;
        private RequestProducer producer;

        private Builder() {
        }

        public Builder setShardId(int shardId) {
            this.shardId = shardId;
            return this;
        }

        public Builder setLocalId(long localId) {
            this.localId = localId;
            return this;
        }

        public Builder setFromMillis(long fromMillis) {
            this.fromMillis = fromMillis;
            return this;
        }

        public Builder setFrom(String time) {
            this.fromMillis = Instant.parse(time).toEpochMilli();
            return this;
        }

        public Builder setToMillis(long toMillis) {
            this.toMillis = toMillis;
            return this;
        }

        public Builder setTo(String time) {
            this.toMillis = Instant.parse(time).toEpochMilli();
            return this;
        }

        public Builder setDeadline(long deadline) {
            this.deadline = deadline;
            return this;
        }

        public Builder setProducer(RequestProducer producer) {
            this.producer = producer;
            return this;
        }

        public StockpileMetricReadRequest build() {
            return new StockpileMetricReadRequest(this);
        }
    }
}
