package ru.yandex.stockpile.server.shard;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.TimeUnit;

import ru.yandex.monlib.metrics.primitives.GaugeInt64;
import ru.yandex.monlib.metrics.primitives.Histogram;
import ru.yandex.solomon.memory.layout.MemMeasurable;
import ru.yandex.solomon.memory.layout.MemoryCounter;
import ru.yandex.solomon.util.collection.queue.ArrayListLockQueue;

/**
 * @author Vladimir Gordiychuk
 */
public class StockpileRequestsQueue<T extends StockpileRequest> implements MemMeasurable {
    private static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(StockpileRequestsQueue.class);

    private final GaugeInt64 queueBytes;
    private final GaugeInt64 queueRequests;
    private final Histogram inQueueElapsedTimeMillis;
    private final ArrayListLockQueue<T> queue = new ArrayListLockQueue<>();

    public StockpileRequestsQueue(GaugeInt64 queueBytes, GaugeInt64 queueRequests, Histogram inQueueElapsedTimeMillis) {
        this.queueBytes = queueBytes;
        this.queueRequests = queueRequests;
        this.inQueueElapsedTimeMillis = inQueueElapsedTimeMillis;
    }


    public void enqueue(T request) {
        queueBytes.add(request.memorySizeIncludingSelf());
        queueRequests.add(1);
        queue.enqueue(request);
    }

    public void enqueueAll(List<T> requests) {
        queueBytes.add(requests.stream().mapToLong(MemMeasurable::memorySizeIncludingSelf).sum());
        queueRequests.add(requests.size());
        queue.enqueueAll(requests);
    }

    public ArrayList<T> dequeueAll() {
        ArrayList<T> result = queue.dequeueAll();
        long bytes = 0;
        long now = System.nanoTime();
        for (T request : result) {
            bytes += request.memorySizeIncludingSelf();
            long duration = now - request.getCreatedAtNanos();
            inQueueElapsedTimeMillis.record(TimeUnit.NANOSECONDS.toMillis(duration));
        }

        queueBytes.add(-bytes);
        queueRequests.add(-result.size());
        return result;
    }

    @Override
    public long memorySizeIncludingSelf() {
        return SELF_SIZE
            + queueBytes.get()
            + ArrayListLockQueue.SELF_SIZE
            + MemoryCounter.arraySize(queueRequests.get(), MemoryCounter.OBJECT_POINTER_SIZE);
    }
}
