package ru.yandex.stockpile.server.shard;

import java.time.Instant;
import java.util.ArrayList;
import java.util.EnumMap;
import java.util.List;
import java.util.Optional;
import java.util.OptionalInt;
import java.util.OptionalLong;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.Executor;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicIntegerFieldUpdater;
import java.util.concurrent.atomic.AtomicReferenceFieldUpdater;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.WillClose;

import com.google.common.base.MoreObjects;
import com.google.common.collect.ImmutableList;
import it.unimi.dsi.fastutil.longs.Long2LongMap;
import it.unimi.dsi.fastutil.longs.Long2LongOpenHashMap;
import it.unimi.dsi.fastutil.longs.Long2ObjectOpenHashMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.misc.actor.ActorRunner;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.misc.lang.EnumMapUtils;
import ru.yandex.misc.lang.Verify;
import ru.yandex.misc.thread.ThreadLocalTimeout;
import ru.yandex.misc.thread.ThreadLocalTimeoutException;
import ru.yandex.misc.thread.WhatThreadDoes;
import ru.yandex.solomon.codec.archive.MetricArchiveImmutable;
import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.codec.archive.header.MetricHeader;
import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.memory.layout.MemMeasurable;
import ru.yandex.solomon.memory.layout.MemMeasurableSubsystem;
import ru.yandex.solomon.memory.layout.MemoryBySubsystem;
import ru.yandex.solomon.memory.layout.MemoryCounter;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.solomon.model.timeseries.AggrGraphDataListIterator;
import ru.yandex.solomon.selfmon.executors.CpuMeasureExecutor;
import ru.yandex.solomon.staffOnly.manager.ExtraContentParam;
import ru.yandex.solomon.staffOnly.manager.find.NamedObject;
import ru.yandex.solomon.staffOnly.manager.special.DurationMillis;
import ru.yandex.solomon.staffOnly.manager.special.ExtraContent;
import ru.yandex.solomon.staffOnly.manager.special.InstantMillis;
import ru.yandex.solomon.staffOnly.manager.special.PullHere;
import ru.yandex.solomon.staffOnly.manager.table.TableColumn;
import ru.yandex.solomon.util.ExceptionUtils;
import ru.yandex.solomon.util.collection.queue.ArrayListLockQueue;
import ru.yandex.stockpile.api.EStockpileStatusCode;
import ru.yandex.stockpile.api.MetricMeta;
import ru.yandex.stockpile.api.grpc.StockpileRuntimeException;
import ru.yandex.stockpile.api.grpc.StockpileShardStopException;
import ru.yandex.stockpile.client.shard.StockpileMetricId;
import ru.yandex.stockpile.client.shard.StockpileShardId;
import ru.yandex.stockpile.kikimrKv.KvStockpileShardStorage;
import ru.yandex.stockpile.server.ListOfOptional;
import ru.yandex.stockpile.server.SnapshotLevel;
import ru.yandex.stockpile.server.Txn;
import ru.yandex.stockpile.server.data.chunk.DataRangeGlobal;
import ru.yandex.stockpile.server.data.chunk.IndexRangeResult;
import ru.yandex.stockpile.server.data.dao.StockpileShardStorageMeasured;
import ru.yandex.stockpile.server.data.index.SnapshotIndex;
import ru.yandex.stockpile.server.data.index.SnapshotIndexContent;
import ru.yandex.stockpile.server.data.index.stats.IndexStatsLevel;
import ru.yandex.stockpile.server.data.log.LogReason;
import ru.yandex.stockpile.server.shard.actor.ActorRunnable;
import ru.yandex.stockpile.server.shard.actor.ActorRunnableType;
import ru.yandex.stockpile.server.shard.actor.InActor;
import ru.yandex.stockpile.server.shard.actor.StockpileShardActState;
import ru.yandex.stockpile.server.shard.cache.MetricDataCache;
import ru.yandex.stockpile.server.shard.cache.MetricDataCacheEntry;
import ru.yandex.stockpile.server.shard.stat.StockpileShardDiskStats;
import ru.yandex.stockpile.server.shard.stat.StockpileShardStats;
import ru.yandex.stockpile.tool.Sampler;

import static java.util.concurrent.CompletableFuture.failedFuture;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class StockpileShard implements NamedObject, MemMeasurableSubsystem {
    private static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(StockpileShard.class);
    private static final Sampler logReadSampler = new Sampler(0.2, 1000);
    private static final Sampler logErrorSampler = new Sampler(1, 10);

    private static final Logger logger = LoggerFactory.getLogger(StockpileShard.class);

    public static final int MAX_METRICS_READ_META = 10_000;

    final ActorRunner actorRunner;
    @TableColumn
    public final int shardId;
    @TableColumn
    public final long kvTabletId;
    public final StockpileShardStorageMeasured storage;
    @InstantMillis
    public final long stockpileShardCreatedInstantMillis;
    final StockpileShardGlobals globals;
    final String debug;
    public final StockpileShardMetrics metrics;
    final Executor commonExecutor;
    final Executor readExecutor;
    final Executor mergeExecutor;

    @PullHere
    final StockpileShardStats stats;
    // DI

    volatile boolean stop = false;
    volatile boolean generationChanged = false;

    public long memoryLimit = 1 << 30; // 1 GiB by default initial limit

    /** Since last snapshot or during startup before logs are loaded */
    @Nonnull
    final LogState logState;
    /**
     *  Null until loading to state when ready to accept write txn
     */
    TxTracker txTracker;
    public final MetricDataCache cache;
    private volatile Future delayedLoad = CompletableFuture.completedFuture(null);

    public StockpileShard(StockpileShardGlobals globals, int shardId, long kvTabletId, @Nonnull String debug) {
        this(globals, shardId, kvTabletId, KvStockpileShardStorage.TABLE_GENERATION_UNKNOWN, debug);
    }

    public StockpileShard(StockpileShardGlobals globals, int shardId, long kvTabletId, long kvTabletGen, @Nonnull String debug) {
        this.globals = globals;
        this.debug = debug;
        StockpileShardId.validate(shardId);
        this.metrics = new StockpileShardMetrics(String.valueOf(shardId));
        this.commonExecutor = new CpuMeasureExecutor(metrics.utimeNanos, globals.executorService);
        this.readExecutor = new CpuMeasureExecutor(metrics.utimeNanos, globals.stockpileReadExecutor);
        this.mergeExecutor = new CpuMeasureExecutor(metrics.utimeNanos, globals.mergeExecutor);
        this.stats = new StockpileShardStats(metrics);

        this.shardId = shardId;
        this.kvTabletId = kvTabletId;
        logState = new LogState(this.shardId);
        cache = new MetricDataCache(this.shardId);
        readQueue = new StockpileRequestsQueue<>(
            metrics.read.queueBytes,
            metrics.read.queueRequests,
            globals.stockpileShardAggregatedStats.readInQueueTimeHistogram);

        // TODO: use factory
        KvStockpileShardStorage
            storage = new KvStockpileShardStorage(globals.readBatcher, globals.kikimrKvClient, kvTabletId, kvTabletGen);
        this.storage = new StockpileShardStorageMeasured(storage);

        this.stockpileShardCreatedInstantMillis = System.currentTimeMillis();
        this.metrics.registry.lazyCounter("stockpile.shard.uptime", () -> System.currentTimeMillis() - stockpileShardCreatedInstantMillis);
        actorRunner = new ActorRunner(this::actOuter, commonExecutor);

        wtd = "shard " + shardId;
        writeProcess = new LogProcess(this);
        allocateIdsProcess = new AllocateLocalIdsProcess(this);
    }

    public StockpileShardStats getStats() {
        return stats;
    }

    public long getGeneration() {
        return storage.getGeneration();
    }

    public StockpileShardMetrics metrics() {
        metrics.memory.set(memoryUsage());
        metrics.memoryLimit.set(memoryLimit);
        metrics.cache.update(cache);
        metrics.disk.update(diskStats());
        metrics.records.set(recordCount().orElse(0));
        metrics.metrics.set(metricsCount().orElse(0));
        return metrics;
    }

    public long memoryUsage() {
        return memorySizeIncludingSelf()
            - cache.memorySizeIncludingSelf()
            - MemMeasurable.memorySizeOfNullable(mergeProcess);
    }

    public boolean isMemoryLimitReach() {
        long usage = memoryUsage();
        if (usage < memoryLimit) {
            return false;
        }

        // TODO: use something else to measure only shard usage
        if (StockpileMemory.getHostMemoryUsage() >= StockpileMemory.getTotalShardsMemoryLimit()) {
            return true;
        }

        long overcommit = usage - memoryLimit;
        return overcommit >= memoryLimit;
    }

    CountDownLatch initializedOrError = new CountDownLatch(1);

    Throwable lastError;
    @InstantMillis
    public long lastErrorInstant;

    public OptionalInt snapshotCount(SnapshotLevel level) {
        return state.snapshotCount(level);
    }

    @Override
    public void addMemoryBySubsystem(MemoryBySubsystem memory) {
        memory.addMemory("stockpile.shard.other", SELF_SIZE);
        memory.addMemory("stockpile.shard.cache", cache.memorySizeIncludingSelf());
        memory.addMemory("stockpile.shard.read.queue", readQueue.memorySizeIncludingSelf());

        for (ProcessType processType : ProcessType.values()) {
            String key = "stockpile_shard_proc_" + processType.name();
            memory.addMemory(key, MemMeasurable.memorySizeOfNullable(process(processType)));
        }

        state.addMemoryBySubsystem(memory);
        storage.addMemoryBySubsystem(memory);
        logState.addMemoryBySubsystem(memory);
    }

    public StockpileShardDiskStats diskStats() {
        return state.diskStats();
    }

    public CompletableFuture<Void> flushCache() {
        return runAsync(ActorRunnableType.FLUSH_CACHE, a -> cache.flush());
    }

    public boolean isOk() {
        // TODO: also check last errors
        return state instanceof StockpileShardStateDone;
    }

    public IndexStatsLevel indexStats() {
        return state.indexesStats();
    }

    @Nullable
    ShardProcess process(ProcessType processType) {
        switch (processType) {
            case LOAD: return loadProcess;
            case ETERNITY_MERGE:
            case DAILY_MERGE:
                MergeProcess mergeProcess = this.mergeProcess;
                if (mergeProcess != null && mergeProcess.mergeKind.processType == processType) {
                    return mergeProcess;
                } else {
                    return null;
                }
            case TWO_HOUR_SNAPSHOT: return twoHourSnapshotProcess;
            case WRITE: return writeProcess;
            case LOG_SNAPSHOT: return logSnapshotProcess;
            case ALLOCATE_LOCAL_IDS: return allocateIdsProcess;
            default: throw new IllegalStateException();
        }
    }

    @Nonnull
    public LoadState getLoadState() {
        return state.loadStateForMon();
    }

    public double getLoadingProgress() {
        var copy = loadProcess;
        if (copy != null) {
            return copy.getLoadingProgress();
        }

        if (getLoadState() == LoadState.DONE) {
            return 100.0;
        } else {
            return 0.0;
        }
    }

    public double getSnapshotProgress(SnapshotLevel level) {
        if (level == SnapshotLevel.TWO_HOURS && twoHourSnapshotProcess != null) {
            return 50.0;
        }

        if (mergeProcess == null || mergeProcess.mergeKind.targetLevel != level) {
            return 100.0;
        }

        return mergeProcess.getProgress();
    }

    public void setMaxCacheSize(long cacheSize) {
        cache.setMaxSizeBytes(cacheSize);
    }

    @Override
    public String namedObjectId() {
        return Integer.toString(shardId);
    }

    public void setMemoryLimit(long memoryLimit) {
        this.memoryLimit = memoryLimit;
    }

    public OptionalLong recordCount() {
        return state.recordCountReliable();
    }

    public OptionalLong metricsCount() {
        return state.recordCountReliable();
    }

    public long minRequiredMemory() {
        return state.minRequiredMemory();
    }

    public enum LoadState {
        INIT,
        LOCKED,
        LOADING,
        LOADING_ERRORS,
        DONE,
        ;
    }


    @Nonnull
    private volatile StockpileShardState state = new StockpileShardStateInit(this);

    private static final AtomicReferenceFieldUpdater<StockpileShard, StockpileShardState> stateField =
        AtomicReferenceFieldUpdater.newUpdater(StockpileShard.class, StockpileShardState.class, "state");

    boolean stateCompareAndSet(StockpileShardState compare, StockpileShardState set) {
        return stateField.compareAndSet(this, compare, set);
    }

    boolean compareAndSetP(Predicate<StockpileShardState> pred, StockpileShardState set) {
        for (;;) {
            StockpileShardState state = this.state;
            if (!pred.test(state)) {
                return false;
            }
            if (stateCompareAndSet(state, set)) {
                return true;
            }
        }
    }

    void stateCheckAndSet(Predicate<StockpileShardState> pred, StockpileShardState set) {
        if (!compareAndSetP(pred, set)) {
            ExceptionUtils.uncaughtException(new RuntimeException("wrong state: " + state));
        }
    }

    public boolean canServeReads() {
        if (stop) {
            return false;
        }

        return getLoadState() == LoadState.DONE;
    }

    public boolean canServeWrites() {
        if (stop) {
            return false;
        }

        return txTracker != null;
    }

    void error(@Nullable Throwable e) {
        if (e == null) {
            return;
        }

        if (logErrorSampler.acquire()) {
            logger.error("error on shard " + StockpileShardId.toString(shardId), e);
        }
        lastError = e;
        lastErrorInstant = System.currentTimeMillis();
        initializedOrError.countDown();
        metrics.errors.inc();
    }

    private final String wtd;

    private void actOuter() {
        long actStartMillis = System.currentTimeMillis();

        try {
            WhatThreadDoes.Handle push = WhatThreadDoes.push(wtd);
            try {
                act();
            } finally {
                push.popSafely();
            }
        } catch (Throwable x) {
            if (stop) {
                logger.error("uncaught exception in shard: {}", shardId, x);
            } else {
                ExceptionUtils.uncaughtException(new RuntimeException("uncaught exception in shard: " + shardId, x));
            }
        }

        metrics.act.count.inc();
        metrics.act.time.add(System.currentTimeMillis() - actStartMillis);
    }

    private void act() {
        if (stop) {
            metrics.act.switchToState(StockpileShardActState.STOPPING);
            for (ProcessType processType : ProcessType.values()) {
                ShardProcess process = process(processType);
                if (process == null) {
                    continue;
                }

                // mostly complete futures
                process.stoppedReleaseResources();
                logState.close();
                cache.close();
            }

            state.stop(InActor.A);
        }

        processRunnables(InActor.A);
        processShardLoad(InActor.A);

        // writes can be served even before initialized
        metrics.act.switchToState(StockpileShardActState.WRITES);
        writeProcess.act(InActor.A);

        metrics.act.switchToState(StockpileShardActState.READS);
        processReadRequests(InActor.A);

        metrics.act.switchToState(StockpileShardActState.FORCE_SNAPSHOT);
        processForceSnapshot(InActor.A);
        metrics.act.switchToState(StockpileShardActState.MERGE);
        for (MergeKind mergeKind : MergeKind.values()) {
            processForceMerge(InActor.A, mergeKind);
        }
        metrics.act.switchToState(StockpileShardActState.RUNNABLE_MISC);
        allocateIdsProcess.act(InActor.A);
        metrics.act.switchToState(StockpileShardActState.SLEEPING);
    }

    private void processShardLoad(InActor a) {
        if (!stop && state instanceof StockpileShardStateInit) {
            StockpileShardStateInit initState = (StockpileShardStateInit) this.state;
            initState.canLoadData()
                    .whenComplete((loadEnabled, e) -> {
                        if (e != null) {
                            error(e);
                            actorRunner.schedule();
                            return;
                        }

                        if (loadEnabled) {
                            run(ActorRunnableType.MISC, actor -> {
                                if (!stop && state instanceof StockpileShardStateInit) {
                                    metrics.act.switchToState(StockpileShardActState.LOAD);
                                    loadProcess = new LoadProcess(this);
                                    loadProcess.start(InActor.A);
                                }
                            });
                        } else if (delayedLoad.isDone()) {
                            long delayMs = TimeUnit.MINUTES.toMillis(1) + ThreadLocalRandom.current().nextLong(15_000);
                            delayedLoad = globals.scheduledExecutorService.schedule(actorRunner::schedule, delayMs, TimeUnit.MILLISECONDS);
                        }
                    });
        }
    }

    void checkSizeStartSnapshot(InActor a) {
        if (logSnapshotProcess != null) {
            return;
        }

        if (twoHourSnapshotProcess != null) {
            return;
        }

        if (isStop()) {
            return;
        }

        if (logState.isEmpty()) {
            return;
        }

        if (checkSizeAndStartTwoHoursSnapshot(a)) {
            return;
        }

        checkSizeAndStartLogCompaction(a);
    }

    private void checkSizeAndStartLogCompaction(InActor a) {
        if (logState.anyReasonToLog() != LogReason.TX) {
            metrics.act.switchToState(StockpileShardActState.LOG_SNAPSHOT);
            logSnapshotProcess = new LogSnapshotProcess(this, txTracker.allocateTx());
            logSnapshotProcess.start(a);
        }
    }

    private boolean checkSizeAndStartTwoHoursSnapshot(InActor a) {
        // two hours snapshot not available at loading state
        if (getLoadState() != LoadState.DONE) {
            return false;
        }

        if (logState.memorySizeIncludingSelf() < memoryLimitForUnflushed()) {
            return false;
        }

        // Do not force snapshots too often even if it is not enough memory.
        // Because otherwize it won't be possible to merge these snapshots.
        // TODO: raise a flag
        if (stateDone().snapshotCount(SnapshotLevel.TWO_HOURS).orElse(0) > 50) {
            return false;
        }

        twoHourSnapshotProcess = new TwoHourSnapshotProcess(stateDone(), List.of(), SnapshotReason.SPACE, txTracker.allocateTx(), a);
        twoHourSnapshotProcess.start(a);
        return true;
    }

    long memoryLimitForUnflushed() {
        long limit = memoryLimit - minRequiredMemory();
        return Math.round(limit * 0.8);
    }

    private void processRunnables(InActor a) {
        ArrayList<ActorRunnable> runnables = runnablesQueue.dequeueAll();
        for (ActorRunnable runnable : runnables) {
            metrics.act.switchToState(runnable.type().state());
            runnable.run(a);
        }
    }


    /**
     * Use by merge process, because it deletes snapshots.
     *
     * 2h snapshot process does not update this field, because it does not delete files.
     */
    private volatile int renameInProgress = 0;

    private static final AtomicIntegerFieldUpdater<StockpileShard> renameInProgressField =
        AtomicIntegerFieldUpdater.newUpdater(StockpileShard.class, "renameInProgress");

    void setRenameInProgress() {
        int prev = renameInProgressField.getAndSet(this, 1);
        Verify.V.isTrue(prev == 0);
    }

    void clearRenameInProgress() {
        int prev = renameInProgressField.getAndSet(this, 0);
        Verify.V.isTrue(prev == 1);
    }


    private void processReadRequests(InActor a) {
        if (!canServeReads()) {
            RuntimeException e = stopException();
            for (StockpileMetricReadRequest readRequest : readQueue.dequeueAll()) {
                logger.info("cannot process request, because shard {} is not ready", shardId);
                readRequest.getFuture().completeExceptionally(e);
            }
            return;
        }

        if (renameInProgress != 0) {
            return;
        }

        ArrayList<StockpileMetricReadRequest> readRequests = readQueue.dequeueAll();

        for (StockpileMetricReadRequest readRequest : readRequests) {
            try {
                ThreadLocalTimeout.Handle handle;
                try {
                    handle = ThreadLocalTimeout.pushInstantMillis(readRequest.getDeadline());
                } catch (Exception e) {
                    readRequest.getFuture().completeExceptionally(e);
                    continue;
                }

                try {
                    processReadRequest(readRequest, a);
                } catch (Exception e) {
                    readRequest.getFuture().completeExceptionally(e);
                } finally {
                    handle.pop();
                }
            } catch (Throwable x) {
                ExceptionUtils.uncaughtException(new RuntimeException(
                    "failed to process " + readRequest.getLocalId() + "; shardId: " + shardId, x));
            }
        }
    }

    private Long2ObjectOpenHashMap<ReadInProgress> waitingRequests = new Long2ObjectOpenHashMap<>();

    public int getWaitingRequestCountForMon() {
        // not thread-safe, but OK for mon
        return waitingRequests.size();
    }

    private String descReadForLog(StockpileMetricReadRequest req) {
        return "read " + StockpileMetricId.toString(shardId, req.getLocalId()) + " as " + req.getProducer();
    }

    private void processReadRequest(StockpileMetricReadRequest readRequest, InActor a) {
        {
            ReadInProgress readInProgress = waitingRequests.get(readRequest.getLocalId());
            if (readInProgress != null && !readInProgress.isDone() && readInProgress.readRangeFromMillis <= readRequest.getFromMillis()) {
                logger.info(descReadForLog(readRequest) + " attaching to late read");
                readInProgress.addLateRead(readRequest);
                return;
            }
        }

        if (cache.isEnabled()) {
            // TODO: snapshot not close at this point, but it's safe right now, because cache not use off-heap cache, or pooled objects
            var snapshot = cache.getSnapshot(readRequest.getLocalId(), readRequest.getFromMillis(), readRequest.getToMillis());
            if (snapshot != null) {
                readExecutor.execute(() -> {
                    WhatThreadDoes.Handle h = WhatThreadDoes.push("Processing cached read: " + readRequest);
                    try {
                        if (isDeadlineExpired(readRequest.getDeadline())) {
                            String message = "Timeout " + Instant.ofEpochMilli(readRequest.getDeadline());
                            readRequest.getFuture().completeExceptionally(new ThreadLocalTimeoutException(message));
                            return;
                        }

                        var ownerId = Integer.toUnsignedString(snapshot.header().getOwnerShardId());
                        if (logReadSampler.acquire()) {
                            logger.info(descReadForLog(readRequest) + " from " + ownerId + " complete " + ReadResultStatus.CACHED);
                        }
                        globals.stockpileShardAggregatedStats.readCompleted(ReadResultStatus.CACHED);

                        StockpileMetricReadResponse response = ReadReply.constructResponse(readRequest, snapshot);
                        completeReadRequest(readRequest, response);
                    } catch (Throwable x) {
                        readRequest.getFuture().completeExceptionally(x);
                    } finally {
                        h.popSafely();
                    }
                });
                return;
            }
            metrics.cache.avgMiss.mark();
        }

        processReadFromStorage(a, readRequest);
    }

    private void completeReadRequest(StockpileMetricReadRequest request, StockpileMetricReadResponse response) {
        metrics.read.avgMetricsRps.mark();
        metrics.read.avgRecordsRps.mark(response.getTimeseries().getRecordCount());
        globals.stockpileShardAggregatedStats.readMetricsRate.inc();
        globals.stockpileShardAggregatedStats.readPointsRate.add(response.getTimeseries().getRecordCount());
        if (response.getTimeseries() instanceof AggrGraphDataArrayList) {
            globals.usage.read(request.getProducer(), response);
        } else {
            var lazy = globals.usage.lazyRead(request.getProducer(), response.getHeader(), response.getTimeseries());
            response = new StockpileMetricReadResponse(response.getHeader(), lazy);
        }
        request.getFuture().complete(response);
    }

    private void processReadFromStorage(InActor a, StockpileMetricReadRequest readRequest) {
        MetricArchiveMutable[] fromMemory = archivesFromMemoryStats(a, readRequest.getLocalId());
        var indexLookup = state.rangeRequestsForMetric(a, readRequest.getLocalId(), readRequest.getFromMillis());
        DataRangeGlobal[] ranges = indexLookup.getRanges();
        long readRangeFromMillis = indexLookup.getFromTimeMillis();
        globals.stockpileShardAggregatedStats.readAmplification.record(ranges.length);

        ReadInProgress readInProgress = new ReadInProgress(readRangeFromMillis, readRequest);
        ReadInProgress prev = waitingRequests.put(readRequest.getLocalId(), readInProgress);
        if (prev != null && !prev.isDone() && prev.readRangeFromMillis <= readRangeFromMillis) {
            waitingRequests.put(readRequest.getLocalId(), prev);
            logger.info(descReadForLog(readRequest) + " attaching to late read");
            prev.addLateRead(readRequest);
            return;
        }

        long startNanos = System.nanoTime();
        final long localId = readRequest.getLocalId();
        logger.info(descReadForLog(readRequest) + " from {} sns", ranges.length);
        storage.readSnapshotRanges(ranges)
                .thenApplyAsync(mas -> mergeReadsFromStorage(localId, indexLookup, mas, fromMemory), readExecutor)
                .whenComplete((response, e) -> {
                    if (ExceptionHandler.isGenerationChanged(e)) {
                        generationChanged = true;
                        stop();
                        run(ActorRunnableType.COMPLETE_READ_FAIL, a2 -> {
                            completeReadFailureInActor(a2, new StockpileRuntimeException(EStockpileStatusCode.SHARD_NOT_READY, "generation changed"), readInProgress);
                        });
                        return;
                    }

                    if (e != null) {
                        run(ActorRunnableType.COMPLETE_READ_FAIL, a2 -> {
                            completeReadFailureInActor(a2, e, readInProgress);
                        });
                        return;
                    }

                    long elapsedNanos = System.nanoTime() - startNanos;
                    globals.stockpileShardAggregatedStats.readMetricFromDiskElapsedTimeMillis.record(TimeUnit.NANOSECONDS.toMillis(elapsedNanos));
                    run(ActorRunnableType.COMPLETE_READ_OK, a2 -> {
                        completeReadInActor(a2, readInProgress, response);
                    });
                });
    }

    private boolean isDeadlineExpired(long deadline) {
        return deadline != 0L && System.currentTimeMillis() >= deadline;
    }

    private boolean killReadRequest(ReadInProgress read) {
        return waitingRequests.remove(read.getLocalId(), read);
    }

    private void logReadFailed(ReadInProgress readInProgress, Throwable x) {
        if (logErrorSampler.acquire()) {
            logger.error(descReadForLog(readInProgress.getInitRequest()) + " failed", x);
        }
    }

    private void completeReadFailureInActor(InActor a, Throwable x, ReadInProgress readInProgress) {
        logReadFailed(readInProgress, x);
        globals.stockpileShardAggregatedStats.readCompleted(ReadResultStatus.ERROR);
        if (killReadRequest(readInProgress)) {
            cache.removeEntry(readInProgress.getLocalId());
        }

        readInProgress.completeExceptionally(x);
    }

    private MetricArchiveMutable[] archivesFromMemoryStats(InActor a, long localId) {
        return memoryState(localId)
            .map(MetricArchiveMutable::new)
            .toArray(MetricArchiveMutable[]::new);
    }

    private void completeReadInActor(
        InActor a,
        ReadInProgress readInProgress,
        @WillClose MetricDataCacheEntry freshCacheEntry)
    {
        try (freshCacheEntry) {
            boolean updateCache = killReadRequest(readInProgress);
            logger.info(descReadForLog(readInProgress.getInitRequest()) + " from disk, attach size {}", readInProgress.reads.size());
            long minFromMillis = readInProgress.reads.stream()
                .mapToLong(StockpileMetricReadRequest::getFromMillis)
                .min()
                .orElse(0L);
            @WillClose
            var snapshot = freshCacheEntry.snapshot(minFromMillis, Long.MAX_VALUE);

            List<StockpileMetricReadRequest> reads = ImmutableList.copyOf(readInProgress.reads);
            readExecutor.execute(() -> {
                WhatThreadDoes.Handle h = WhatThreadDoes.push("Processing disk read: " + StockpileMetricId
                    .toString(reads.get(0).getShardId(), reads.get(0).getLocalId()));
                try {
                    AtomicInteger cnt = new AtomicInteger(reads.size());
                    for (int index = 1; index < reads.size(); index++) {
                        StockpileMetricReadRequest request = reads.get(index);
                        readExecutor.execute(() -> {
                            try {
                                completeUserReadRequest(request, snapshot.header(), snapshot.iterator());
                            } finally {
                                if (cnt.decrementAndGet() == 0) {
                                    snapshot.release();
                                }
                            }
                        });
                    }

                    completeUserReadRequest(reads.get(0), snapshot.header(), snapshot.iterator());
                    if (cnt.decrementAndGet() == 0) {
                        snapshot.release();
                    }
                } catch (Throwable e) {
                    logger.error(descReadForLog(reads.get(0)) + " fail cache update ", e);
                    for (StockpileMetricReadRequest request : reads) {
                        request.getFuture().completeExceptionally(e);
                    }
                } finally {
                    h.popSafely();
                }
            });

            if (cache.isEnabled() && updateCache) {
                cache.readCompleted(readInProgress.getLocalId(), freshCacheEntry);
            }
        } catch (Throwable e) {
            logger.error(descReadForLog(readInProgress.getInitRequest()) + " fail cache update", e);
            readInProgress.completeExceptionally(e);
        }
    }

    private void completeUserReadRequest(StockpileMetricReadRequest request, MetricHeader header, AggrGraphDataListIterator it) {
        CompletableFuture<StockpileMetricReadResponse> future = request.getFuture();
        try {
            if (isDeadlineExpired(request.getDeadline())) {
                String message = "Timeout " + Instant.ofEpochMilli(request.getDeadline());
                future.completeExceptionally(new ThreadLocalTimeoutException(message));
                return;
            }

            completeReadRequest(request, ReadReply.constructResponse(request, header, it));
        } catch (Throwable e) {
            future.completeExceptionally(e);
        }
    }

    private MetricDataCacheEntry mergeReadsFromStorage(long localId, IndexRangeResult indexLockup, ListOfOptional<MetricArchiveImmutable> archivesFromDiskWithNulls, MetricArchiveMutable[] fromMemory) {
        WhatThreadDoes.Handle h = WhatThreadDoes.push("Merge disk read: " + StockpileMetricId.toString(shardId, localId));
        try {
            // https://st.yandex-team.ru/SOLOMON-1055
            if (archivesFromDiskWithNulls.anyNulls()) {
                throw new RuntimeException("database corrupted; shard: " + shardId);
            }

            MetricArchiveImmutable[] fromDisk = archivesFromDiskWithNulls.getAllNoNulls();
            ReadResultStatus readResultStatus;
            if (fromDisk.length == 0) {
                readResultStatus = ReadResultStatus.FROM_STORAGE_MEM_ONLY;
            } else {
                readResultStatus = ReadResultStatus.FROM_STORAGE;
            }
            globals.stockpileShardAggregatedStats.readCompleted(readResultStatus);

            var combiner = new ArchiveCombiner(shardId, localId);
            for (int index = 0; index < archivesFromDiskWithNulls.size(); index++) {
                var archive = archivesFromDiskWithNulls.getOrNull(index);
                if (archive != null) {
                    combiner.add(archive, indexLockup.getEntries().get(index).getLastTsMillis());
                }
            }

            for (var memory : fromMemory) {
                combiner.add(memory);
            }

            var combined = combiner.combine();
            return MetricDataCacheEntry.of(indexLockup.getFromTimeMillis(), combined.getHeader(), combined.getItemIterator());
        } finally {
            for (var archive : fromMemory) {
                archive.close();
            }
            h.popSafely();
        }
    }

    private Stream<MetricArchiveMutable> memoryState(long localId) {
        if (twoHourSnapshotProcess != null) {
            return Stream.concat(
                twoHourSnapshotProcess.getMemoryState(localId),
                logState.contentSinceLastSnapshotStream(localId));
        } else {
            return logState.contentSinceLastSnapshotStream(localId);
        }
    }

    @Nonnull
    StockpileShardStateDone stateDone() {
        return (StockpileShardStateDone) state;
    }

    RuntimeException stopException() {
        if (generationChanged) {
            String msg = "generation changed; shard: " + shardId;
            return new StockpileShardStopException(EStockpileStatusCode.SHARD_NOT_READY, msg);
        } else if (stop) {
            return new StockpileShardStopException(EStockpileStatusCode.SHARD_NOT_READY, "stop; shard: " + shardId);
        }

        String msg = "state: " + state.loadStateDescForMon() + "; shard: " + shardId;
        if (lastError != null) {
            msg += "; lastError: " + lastError.getMessage();
        }
        return new StockpileShardStopException(EStockpileStatusCode.SHARD_NOT_READY, msg);
    }

    private void processForceMerge(InActor a, MergeKind mergeKind) {
        if (mergeProcess != null) {
            mergeProcess.act(a);
        }

        ArrayListLockQueue<CompletableFuture<Txn>> forceMergeQueue = forceMergeQueueByMergeKind.get(mergeKind);

        if (stop || !canServeReads()) {
            RuntimeException e = stopException();
            for (CompletableFuture<?> future : forceMergeQueue.dequeueAll()) {
                future.completeExceptionally(e);
            }

            return;
        }

        // Do not do more than once thing in parallel
        if (isMergeOrSnapshotInProgress()) {
            return;
        }

        ArrayList<CompletableFuture<Txn>> forceMergeFutures = forceMergeQueue.dequeueAll();
        if (forceMergeFutures.isEmpty()) {
            return;
        }

        mergeProcess = new MergeProcess(stateDone(), forceMergeFutures, mergeKind, globals.mergeProcessMetrics, a);

        mergeProcess.start(a);
    }

    TwoHourSnapshotProcess twoHourSnapshotProcess;
    MergeProcess mergeProcess;
    LoadProcess loadProcess;
    LogProcess writeProcess;
    LogSnapshotProcess logSnapshotProcess;
    AllocateLocalIdsProcess allocateIdsProcess;


    // randomize schedule period https://st.yandex-team.ru/SOLOMON-1064
    @DurationMillis
    long snapshotPeriodMillis = SnapshotAndMergeScheduler.randomSnapshotPeriod();

    public boolean isMergeOrSnapshotInProgress() {
        return twoHourSnapshotProcess != null || mergeProcess != null;
    }

    private void processForceSnapshot(InActor a) {
        if (twoHourSnapshotProcess != null) {
            twoHourSnapshotProcess.act(a);
        }

        if (stop || !canServeReads()) {
            RuntimeException e = stopException();
            for (CompletableFuture<?> future : forceSnapshotQueue.dequeueAll()) {
                future.completeExceptionally(e);
            }

            return;
        }

        if (logSnapshotProcess != null) {
            return;
        }

        // Do not do more than once thing in parallel
        if (twoHourSnapshotProcess != null) {
            return;
        }

        ArrayList<CompletableFuture<Long>> forceSnapshotFutures = forceSnapshotQueue.dequeueAll();
        if (forceSnapshotFutures.isEmpty()) {
            return;
        }

        if (logState.isEmpty()) {
            stateDone().indexes.updateLatestSnapshotTime(SnapshotLevel.TWO_HOURS, System.currentTimeMillis());
            for (var future: forceSnapshotFutures) {
                future.complete(Txn.ZERO);
            }
            return;
        }

        twoHourSnapshotProcess = new TwoHourSnapshotProcess(stateDone(), forceSnapshotFutures, SnapshotReason.TIME, txTracker.allocateTx(), a);

        twoHourSnapshotProcess.start(a);
    }

    private final StockpileRequestsQueue<StockpileMetricReadRequest> readQueue;
    private ArrayListLockQueue<CompletableFuture<Long>> forceSnapshotQueue = new ArrayListLockQueue<>();
    private EnumMap<MergeKind, ArrayListLockQueue<CompletableFuture<Txn>>> forceMergeQueueByMergeKind =
        EnumMapUtils.fill(MergeKind.class, e -> new ArrayListLockQueue<CompletableFuture<Txn>>());
    private ArrayListLockQueue<ActorRunnable> runnablesQueue = new ArrayListLockQueue<>();

    void run(ActorRunnableType type, Consumer<InActor> fn) {
        runnablesQueue.enqueue(new ActorRunnable() {
            @Override
            public ActorRunnableType type() {
                return type;
            }

            @Override
            public void run(InActor a) {
                fn.accept(a);
            }
        });
        actorRunner.schedule();
    }

    CompletableFuture<Void> runAsync(ActorRunnableType type, Consumer<InActor> fn) {
        CompletableFuture<Void> r = new CompletableFuture<>();
        run(type, a -> {
            try {
                fn.accept(a);
                r.complete(null);
            } catch (Throwable x) {
                r.completeExceptionally(x);
            }
        });
        return r;
    }

    <A> CompletableFuture<A> supplyAsync(ActorRunnableType type, Function<InActor, A> supplier) {
        CompletableFuture<A> r = new CompletableFuture<>();
        run(type, a -> {
            try {
                r.complete(supplier.apply(a));
            } catch (Throwable t) {
                r.completeExceptionally(t);
            }
        });
        return r;
    }

    public CompletableFuture<Txn> pushBatch(StockpileWriteRequest request) {
        if (!request.isEmpty()) {
            writeProcess.enqueue(request);
            actorRunner.schedule();
        } else {
            request.getFuture().completeExceptionally(new StockpileRuntimeException(EStockpileStatusCode.INVALID_REQUEST, "write without points", false));
        }
        return request.getFuture();
    }

    public CompletableFuture<long[]> allocateLocalIds(int size, long deadline) {
        return allocateIdsProcess.allocateLocalIds(size, deadline);
    }

    public CompletableFuture<Txn> forceLogSnapshot() {
        CompletableFuture<Txn> future = new CompletableFuture<>();
        run(ActorRunnableType.MISC, actor -> {
            logState.requestForceLogSnapshot();
            checkSizeStartSnapshot(actor);
            LogSnapshotProcess activeProcess = logSnapshotProcess;
            if (activeProcess != null) {
                CompletableFutures.whenComplete(activeProcess.getFuture(), future);
            } else {
                future.complete(null);
            }
        });
        return future;
    }

    public CompletableFuture<StockpileMetricReadResponse> readOne(StockpileMetricReadRequest request) {
        readQueue.enqueue(request);
        actorRunner.schedule();
        return request.getFuture();
    }

    public void read(List<StockpileMetricReadRequest> requests) {
        readQueue.enqueueAll(requests);
        actorRunner.schedule();
    }

    public CompletableFuture<List<MetricMeta>> readMetricsMeta(long[] localIds) {
        if (localIds.length > MAX_METRICS_READ_META) {
            String msg = "too many metric ids are given, max: " + MAX_METRICS_READ_META;
            return failedFuture(new IllegalArgumentException(msg));
        }

        LoadState state = getLoadState();
        if (getLoadState() != state) {
            String msg = "shard in state: " + state;
            return failedFuture(new StockpileShardStateException(EStockpileStatusCode.SHARD_NOT_READY, msg));
        }

        return supplyAsync(ActorRunnableType.READ_METRICS_META, a -> stateDone()
            .indexes(a)
            .collect(Collectors.toList()))
            .thenApplyAsync(indexes -> {
                Long2LongOpenHashMap found = new Long2LongOpenHashMap(localIds.length);
                found.defaultReturnValue(-1);

                // TODO: optimize
                for (SnapshotIndex index : indexes) {
                    SnapshotIndexContent content = index.getContent();
                    for (long localId : localIds) {
                        long lastTsMillis = content.findMetricLastTsMillis(localId);
                        if (lastTsMillis != -1) {
                            long prevTsMillis = found.put(localId, lastTsMillis);
                            if (prevTsMillis != -1 && prevTsMillis > lastTsMillis) {
                                // UNLIKELY:
                                //     upper index level has older data than lower one,
                                //     so revert back last insert
                                found.put(localId, prevTsMillis);
                            }
                        }
                    }
                }

                List<MetricMeta> metricMetas = new ArrayList<>(found.size());
                MetricMeta.Builder builder = MetricMeta.newBuilder(); // reuse builder
                for (Long2LongMap.Entry e : found.long2LongEntrySet()) {
                    metricMetas.add(builder
                        .setLocalId(e.getLongKey())
                        .setLastTsMillis(e.getLongValue())
                        .build());
                }
                return metricMetas;
        }, commonExecutor);
    }

    public CompletableFuture<Long> forceSnapshot() {
        CompletableFuture<Long> r = new CompletableFuture<>();
        forceSnapshotQueue.enqueue(r);
        actorRunner.schedule();
        return r;
    }

    public long forceSnapshotSync() {
        return CompletableFutures.join(forceSnapshot());
    }

    public CompletableFuture<Txn> forceMerge(MergeKind mergeKind) {
        CompletableFuture<Txn> r = new CompletableFuture<>();
        forceMergeQueueByMergeKind.get(mergeKind).enqueue(r);
        actorRunner.schedule();
        return r;
    }

    public Txn forceMergeSync(MergeKind mergeKind) {
        return CompletableFutures.join(forceMerge(mergeKind));
    }

    @Nonnull
    public SnapshotTs latestSnapshotTime(@Nonnull SnapshotLevel snapshotLevel) {
        return state.latestStapshotTime(snapshotLevel);
    }

    public boolean isGenerationChanged() {
        return generationChanged;
    }

    public void start() {
        actorRunner.schedule();
    }

    public void stop() {
        String stopReason = generationChanged ? "generation changed" : "unknown";
        logger.info("stop shard {}, kvTabletId {}, reason {}", shardId, kvTabletId, stopReason);
        stop = true;
        actorRunner.schedule();
        initializedOrError.countDown();
        // should probably be sync
    }

    public void waitForInitializedOrAnyError() {
        try {
            actorRunner.schedule();
            initializedOrError.await();
        } catch (InterruptedException e) {
            throw new RuntimeException(e);
        }
        if (lastError != null) {
            throw new RuntimeException("failed to initialize", lastError);
        }
    }

    public StockpileShardInspector inspector() {
        return new StockpileShardInspector(this);
    }

    public Optional<StockpileFormat> oldestUsedFormatForMon() {
        return state.oldestUsedFormatForMon();
    }

    @ExtraContent("Info")
    private void extra(ExtraContentParam p) {
        StockpileShardExtra.extra(p, state);
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
            .add("shardId", shardId)
            .add("kvTabletId", kvTabletId)
            .add("loadState", state.loadStateDescForMon())
            .toString();
    }

    public boolean isMerge(MergeKind mergeKind) {
        MergeProcess mergeProcess = this.mergeProcess;
        return mergeProcess != null && mergeProcess.mergeKind == mergeKind;
    }

    public boolean isStop() {
        return stop;
    }
}
