package ru.yandex.stockpile.server.shard;

import java.time.Instant;
import java.util.concurrent.CompletableFuture;

import javax.annotation.Nonnull;

import it.unimi.dsi.fastutil.longs.LongArrayList;
import it.unimi.dsi.fastutil.longs.LongOpenHashSet;

import ru.yandex.misc.lang.StringUtils;
import ru.yandex.salmon.proto.StockpileCanonicalProto;
import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.staffOnly.annotations.ManagerMethod;
import ru.yandex.solomon.staffOnly.annotations.ManagerMethodArgument;
import ru.yandex.solomon.staffOnly.manager.find.NamedObject;
import ru.yandex.stockpile.client.shard.StockpileLocalId;
import ru.yandex.stockpile.ser.test.convert.ArchiveConverters;
import ru.yandex.stockpile.server.shard.actor.ActorRunnableType;
import ru.yandex.stockpile.server.shard.cache.MetricSnapshot;


/**
 * @author Stepan Koltsov
 */
public class StockpileShardInspector implements NamedObject {
    @Nonnull
    private final StockpileShard shard;

    public StockpileShardInspector(@Nonnull StockpileShard shard) {
        this.shard = shard;
    }

    public CompletableFuture<LongOpenHashSet> allLocalIds() {
        return shard.supplyAsync(ActorRunnableType.READ_LOCAL_IDS, a -> {
            LongOpenHashSet r = new LongOpenHashSet();
            shard.stateDone().indexes(a).flatMapToLong(i -> i.getContent().metricIdStream()).forEach(r::add);
            r.addAll(LongArrayList.wrap(shard.logState.metricIds()));
            return r;
        });
    }

    @ManagerMethod
    public CompletableFuture<StockpileCanonicalProto.Archive> getCacheEntryProto(@ManagerMethodArgument(name = "localId") String id) {
        long localId = StockpileLocalId.parse(id);
        return shard
            .supplyAsync(ActorRunnableType.READ_CACHE_ENTRY, a -> processCacheEntryRequest(shard, localId, 0, Long.MAX_VALUE))
            .thenApply(ArchiveConverters::toProto);
    }

    @ManagerMethod
    public CompletableFuture<StockpileCanonicalProto.Archive> getCacheEntryProto(
        @ManagerMethodArgument(name = "localId") String id,
        @ManagerMethodArgument(name = "From UTC instant time(e.g 2019-01-21T07:29:00Z)") String from,
        @ManagerMethodArgument(name = "To UTC instant time(e.g 2019-01-21T07:35:00Z)") String to)
    {
        long localId = StockpileLocalId.parse(id);
        long fromMillis = StringUtils.isEmpty(from) ? 0 : Instant.parse(from).toEpochMilli();
        long toMillis = StringUtils.isEmpty(to) ? Long.MAX_VALUE : Instant.parse(to).toEpochMilli();
        return shard
                .supplyAsync(ActorRunnableType.READ_CACHE_ENTRY, a -> processCacheEntryRequest(shard, localId, fromMillis, toMillis))
                .thenApply(ArchiveConverters::toProto);
    }

    public CompletableFuture<MetricArchiveMutable> readFromDisk(String id) {
        long localId = StockpileLocalId.parse(id);
        // TODO: implement different method to read data from KV-tablet directly,
        //       without affecting shard read queue and placing result into cache
        return shard.readOne(StockpileMetricReadRequest.newBuilder()
                .setLocalId(localId)
                .setDeadline(System.currentTimeMillis() + 120_000)
                .build())
                .thenApply(response -> {
                    MetricArchiveMutable archive = new MetricArchiveMutable(response.getHeader());
                    archive.addAll(response.getTimeseries());
                    return archive;
                });
    }

    @ManagerMethod
    public CompletableFuture<StockpileCanonicalProto.Archive> getDiskEntryProto(String id) {
        return readFromDisk(id).thenApply(ArchiveConverters::toProto);
    }

    private MetricSnapshot processCacheEntryRequest(StockpileShard shard, long localId, long fromMillis, long toMillis) {
        if (shard.cache.isEnabled()) {
            var snapshot = shard.cache.getSnapshot(localId, fromMillis, toMillis);
            if (snapshot == null) {
                throw new RuntimeException("No entry in cache!");
            } else {
                return snapshot;
            }
        } else {
            throw new RuntimeException("Cache is disabled!");
        }
    }

    @Override
    public String namedObjectId() {
        return shard.namedObjectId();
    }
}
