package ru.yandex.stockpile.server.shard;

import java.util.Arrays;
import java.util.Comparator;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Import;
import org.springframework.stereotype.Component;

import ru.yandex.solomon.staffOnly.annotations.LinkedOnRootPage;
import ru.yandex.solomon.staffOnly.annotations.ManagerMethod;
import ru.yandex.solomon.staffOnly.manager.ExtraContentParam;
import ru.yandex.solomon.staffOnly.manager.special.ExtraContent;
import ru.yandex.stockpile.server.shard.stat.StockpileShardAggregatedStats;

/**
 * @author Stepan Koltsov
 */
@Component
@Import({
    StockpileShardGlobals.class,
    StockpileShardAggregatedStats.class,
    StockpileShardLoaderMetricsProvider.class,
    StockpileLocalShards.class
})
@LinkedOnRootPage("Shard Loader")
public class StockpileShardLoader {
    private final StockpileLocalShards shards;

    @Autowired
    public StockpileShardLoader(StockpileLocalShards shards) {
        this.shards = shards;
    }

    @ExtraContent("Local shard stats")
    private void extra(ExtraContentParam p) {
        StockpileShardLoaderWww.extra(p, shards.stream().collect(Collectors.toList()));
    }

    private final AtomicInteger forceSnapshotsInProgress = new AtomicInteger(0);

    @ManagerMethod
    protected String forceShardsLogsSnapshot(long logSizeThresholdMb) {
        int inProgressCount = forceSnapshotsInProgress.get();
        if (inProgressCount != 0) {
            return inProgressCount + " force snapshots already in progress";
        }

        final long logSizeThreshold = logSizeThresholdMb * (1 << 20);
        final int maxShardsCount = 100;

        Comparator<StockpileShard> bySizeAsc =
            Comparator.comparingLong(shard -> shard.logState.diskStats().size());

        StockpileShard[] localShards = shards.stream()
            .filter(shard -> shard.logState.diskStats().size() >= logSizeThreshold)
            .sorted(bySizeAsc.reversed())
            .limit(maxShardsCount)
            .toArray(StockpileShard[]::new);

        forceSnapshotsInProgress.set(localShards.length);

        CompletableFuture[] futures = Arrays.stream(localShards)
            .map(StockpileShard::forceSnapshot)
            .map(f -> f.whenComplete((r, t) -> {
                forceSnapshotsInProgress.decrementAndGet();
            }))
            .toArray(CompletableFuture[]::new);

        return "scheduled " + futures.length + " force snapshots";
    }
}
