package ru.yandex.stockpile.server.shard;

import java.util.Optional;
import java.util.concurrent.TimeUnit;

import javax.annotation.ParametersAreNonnullByDefault;

import it.unimi.dsi.fastutil.objects.Object2IntMap;
import it.unimi.dsi.fastutil.objects.Object2IntOpenHashMap;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.monlib.metrics.MetricConsumer;
import ru.yandex.monlib.metrics.MetricSupplier;
import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.histogram.HistogramCollector;
import ru.yandex.monlib.metrics.histogram.HistogramSnapshot;
import ru.yandex.monlib.metrics.histogram.Histograms;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.util.collection.enums.EnumMapToLong;
import ru.yandex.stockpile.server.SnapshotLevel;
import ru.yandex.stockpile.server.data.names.FileKind;
import ru.yandex.stockpile.server.shard.stat.SizeAndCount;
import ru.yandex.stockpile.server.shard.stat.StockpileShardDiskStats;

/**
 * @author Stepan Koltsov
 */
@Component
@ParametersAreNonnullByDefault
public class StockpileShardLoaderMetricsProvider implements MetricSupplier {
    private final StockpileLocalShards shards;

    @Autowired
    public StockpileShardLoaderMetricsProvider(StockpileLocalShards shards) {
        this.shards = shards;
    }

    @Override
    public int estimateCount() {
        return 20;
    }

    @Override
    public void append(long tsMillis, Labels commonLabels, MetricConsumer consumer) {
        Stat stat = new Stat(tsMillis, commonLabels);
        shards.stream().forEach(stat::add);
        stat.append(consumer);
    }

    private static class Stat {
        private final long tsMillis;
        private final Labels commonLabels;

        private final long now;

        private long shardsCount;
        private long readWaitingDisk;
        private Object2IntOpenHashMap<Optional<StockpileFormat>> shardCountByFormat = new Object2IntOpenHashMap<>();
        private StockpileShardDiskStats diskStats = StockpileShardDiskStats.zeroes();
        private HistogramCollector twoHoursSnapshotCount = Histograms.exponential(16, 2, 1);
        private HistogramCollector dailySnapshotCount = Histograms.exponential(16, 2, 1);
        private HistogramCollector eternitySnapshotCount = Histograms.exponential(16, 2, 1);
        private HistogramCollector daysSinceLastEternitySnapshot = Histograms.exponential(6, 2, 1);
        private HistogramCollector hoursSinceLastDailySnapshot = Histograms.exponential(10, 2, 1);
        private HistogramCollector minutesSinceLastTwoHoursSnapshot = Histograms.exponential(14, 2, 1);
        private EnumMapToLong<ProcessType> processErrors = new EnumMapToLong<>(ProcessType.class);
        private EnumMapToLong<ProcessType> processActive = new EnumMapToLong<>(ProcessType.class);
        private EnumMapToLong<StockpileShard.LoadState> countByState = new EnumMapToLong<>(StockpileShard.LoadState.class);

        Stat(long tsMillis, Labels commonLabels) {
            this.tsMillis = tsMillis;
            this.commonLabels = commonLabels;
            this.now = System.currentTimeMillis();
        }

        public void add(StockpileShard shard) {
            shardsCount++;
            readWaitingDisk = shard.getWaitingRequestCountForMon();
            shardCountByFormat.addTo(shard.oldestUsedFormatForMon(), 1);
            diskStats = StockpileShardDiskStats.add(diskStats, shard.diskStats());
            shard.snapshotCount(SnapshotLevel.TWO_HOURS).ifPresent(value -> twoHoursSnapshotCount.collect(value));
            shard.snapshotCount(SnapshotLevel.DAILY).ifPresent(value -> dailySnapshotCount.collect(value));
            shard.snapshotCount(SnapshotLevel.ETERNITY).ifPresent(value -> eternitySnapshotCount.collect(value));
            addSnapshotStats(shard);
            for (ProcessType processType : ProcessType.values()) {
                ShardProcess process = shard.process(processType);
                if (process != null) {
                    processActive.addAndGet(processType, 1);

                    if (process.lastError != null) {
                        processErrors.addAndGet(processType, 1);
                    }
                }
            }

            countByState.addAndGet(shard.getLoadState(), 1);
        }

        private void addSnapshotStats(StockpileShard shard) {
            long lastEternity = shard.latestSnapshotTime(SnapshotLevel.ETERNITY).getTsOrSpecial();
            if (lastEternity > 0) {
                daysSinceLastEternitySnapshot.collect(TimeUnit.MILLISECONDS.toDays(now - lastEternity));
            }
            long lastDaily = shard.latestSnapshotTime(SnapshotLevel.DAILY).getTsOrSpecial();
            if (lastDaily > 0) {
                hoursSinceLastDailySnapshot.collect(TimeUnit.MILLISECONDS.toHours(now - lastDaily));
            }
            long lastTwoHours = shard.latestSnapshotTime(SnapshotLevel.TWO_HOURS).getTsOrSpecial();
            if (lastTwoHours > 0) {
                minutesSinceLastTwoHoursSnapshot.collect(TimeUnit.MILLISECONDS.toMinutes(now - lastTwoHours));
            }
        }

        public void append(MetricConsumer consumer) {
            append("stockpile.host.shard.count", shardsCount, consumer);
            append("stockpile.host.read.request.disk.waiting", readWaitingDisk, consumer);

            for (Object2IntMap.Entry<Optional<StockpileFormat>> e : shardCountByFormat.object2IntEntrySet()) {
                Labels labels = Labels.of("format", e.getKey().map(Enum::name).orElse("UNKNOWN"));
                append("stockpile.host.shard.format", labels, e.getIntValue(), consumer);
            }

            for (FileKind kind : FileKind.values()) {
                SizeAndCount sizeAndCount = diskStats.get(kind);
                Labels labels = Labels.of("type", kind.monKey);

                append("stockpile.host.file.count", labels, sizeAndCount.count(), consumer);
                append("stockpile.host.file.bytes", labels, sizeAndCount.size(), consumer);
            }

            append("stockpile.host.last.eternity.snapshot.days", daysSinceLastEternitySnapshot.snapshot(), consumer);
            append("stockpile.host.last.daily.snapshot.hours", hoursSinceLastDailySnapshot.snapshot(), consumer);
            append("stockpile.host.last.twoHours.snapshot.minutes", minutesSinceLastTwoHoursSnapshot.snapshot(), consumer);
            append("stockpile.host.twoHours.snapshot.count", twoHoursSnapshotCount.snapshot(), consumer);
            append("stockpile.host.daily.snapshot.count", dailySnapshotCount.snapshot(), consumer);
            append("stockpile.host.eternity.snapshot.count", eternitySnapshotCount.snapshot(), consumer);

            for (ProcessType type : ProcessType.values()) {
                Labels labels = Labels.of("type", type.name());
                append("stockpile.host.process.active", labels, processActive.get(type), consumer);
                append("stockpile.host.process.errors", labels, processErrors.get(type), consumer);
            }

            for (StockpileShard.LoadState state : StockpileShard.LoadState.values()) {
                append("stockpile.shard.state", Labels.of("state", state.name()), countByState.get(state), consumer);
            }
        }

        private void append(String metric, long value, MetricConsumer consumer) {
            append(metric, Labels.empty(), value, consumer);
        }

        private void append(String metric, Labels labels, long value, MetricConsumer consumer) {
            consumer.onMetricBegin(MetricType.IGAUGE);
            consumer.onLabelsBegin(commonLabels.size() + labels.size() + 1);
            commonLabels.forEach(consumer::onLabel);
            labels.forEach(consumer::onLabel);
            consumer.onLabel("sensor", metric);
            consumer.onLabelsEnd();
            consumer.onLong(tsMillis, value);
            consumer.onMetricEnd();
        }

        private void append(String metric, HistogramSnapshot value, MetricConsumer consumer) {
            append(metric, Labels.empty(), value, consumer);
        }

        private void append(String metric, Labels labels, HistogramSnapshot value, MetricConsumer consumer) {
            consumer.onMetricBegin(MetricType.HIST);
            consumer.onLabelsBegin(commonLabels.size() + labels.size() + 1);
            labels.forEach(consumer::onLabel);
            commonLabels.forEach(consumer::onLabel);
            consumer.onLabel("sensor", metric);
            consumer.onLabelsEnd();
            consumer.onHistogram(0, value);
            consumer.onMetricEnd();
        }
    }
}
