package ru.yandex.stockpile.server.shard;

import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import ru.yandex.bolts.collection.CollectorsF;
import ru.yandex.misc.random.Random2;
import ru.yandex.solomon.staffOnly.html.HtmlWriterWithCommonLibraries;
import ru.yandex.solomon.staffOnly.manager.ExtraContentParam;
import ru.yandex.solomon.staffOnly.manager.ManagerController;
import ru.yandex.solomon.staffOnly.manager.table.TableColumnDefImpl;
import ru.yandex.stockpile.server.SnapshotLevel;

/**
 * @author Vladimir Gordiychuk
 */
public class StockpileShardLoaderWww {

    public static void extra(ExtraContentParam p, List<StockpileShard> targets) {
        var shardsByState = targets.stream()
            .collect(Collectors.groupingBy(StockpileShard::getLoadState));

        HtmlWriterWithCommonLibraries hw = p.getHtmlWriter();

        p.managerWriter().listTable(Arrays.asList(StockpileShard.LoadState.values()),
            new TableColumnDefImpl<>("State", s -> s),
            new TableColumnDefImpl<>("Count", s -> shardsByState.getOrDefault(s, List.of()).size()),
            new TableColumnDefImpl<>("Random", s -> {
                List<StockpileShard> shards = shardsByState.getOrDefault(s, List.of());
                if (shards.isEmpty()) {
                    return "";
                } else {
                    StockpileShard randomShard = Random2.threadLocal().randomElement(shards);
                    return ManagerController.objectAHref(randomShard);
                }
            }));

        ShardWithStats[] shards = targets.stream()
            .map(ShardWithStats::new)
            .toArray(ShardWithStats[]::new);

        if (shards.length > 0) {
            class ShardWriter {
                void writeTrThs() {
                    hw.trThsNowrap(
                        "",
                        "Shard",
                        "2h sns", "Recs", "Wr recs/s", "Rds", "Dl ts", "Et ts", "Act cpu", "Act/s", "Wr tx/s");
                }

                void writeShardRow(String property, @Nullable ShardWithStats shard) {
                    hw.tr(() -> {
                        hw.thText(property);
                        if (shard != null) {
                            hw.td(() -> {
                                String href = ManagerController.namedObjectLink(shard.shard.namedObjectIdGlobal());
                                hw.aHref(href, shard.shard.shardId);
                            });
                            hw.tdText(shard.twoHourSnapshots);
                            hw.tdText(shard.recordCount);
                            hw.tdText(String.format("%.1f", shard.writeRecordsPerSecond));
                            hw.tdText(shard.waitingReadRequests);
                            hw.tdText(shard.dailyMergeInstant.toStringToSeconds());
                            hw.tdText(shard.eternityMergeInstant.toStringToSeconds());
                            hw.tdText(String.format("%.3f", shard.actRatio));
                            hw.tdText(String.format("%.3f", shard.actPerSecond));
                            hw.tdText(String.format("%.3f", shard.writeTxsPerSecond));
                        } else {
                            hw.tdColspan(10, () -> hw.write("no shard"));
                        }
                    });
                }
            }

            ShardWithStats largestBySnapshots = Arrays.stream(shards)
                .max(Comparator.comparing(s -> s.twoHourSnapshots))
                .get();

            ShardWithStats largestByRecords = Arrays.stream(shards)
                .max(Comparator.comparing(s -> s.recordCount))
                .get();

            ShardWithStats largestByActRatio = Arrays.stream(shards)
                .max(Comparator.comparingDouble(s -> s.actRatio))
                .get();

            MinMax<ShardWithStats> byActsPerSecond = minMax(
                shards, Comparator.comparingDouble(s -> s.actPerSecond));

            MinMax<ShardWithStats> byWriteTxsPerSecond = minMax(
                shards, Comparator.comparingDouble(s -> s.writeTxsPerSecond));

            MinMax<ShardWithStats> byWriteRecordsPerSecond = minMax(
                shards, Comparator.comparingDouble(s -> s.writeRecordsPerSecond));

            ShardWithStats oldestDailyMerge = Arrays.stream(shards)
                .min(Comparator.comparing(s -> s.dailyMergeInstant.getTsOrSpecial()))
                .get();

            ShardWithStats oldestEternityMerge = Arrays.stream(shards)
                .min(Comparator.comparing(s -> s.eternityMergeInstant.getTsOrSpecial()))
                .get();

            ShardWithStats oldestSnapshot = Arrays.stream(shards)
                .min(Comparator.comparing(s -> s.snapshotInstant.getTsOrSpecial()))
                .get();

            Optional<ShardWithStats> dailyMergeShard = Arrays.stream(shards)
                .filter(s -> s.shard.isMerge(MergeKind.DAILY))
                .findAny();

            Optional<ShardWithStats> eternityMergeShard = Arrays.stream(shards)
                .filter(s -> s.shard.isMerge(MergeKind.ETERNITY))
                .findAny();

            Optional<ShardWithStats> snapshotShard = Arrays.stream(shards)
                .filter(s -> s.shard.twoHourSnapshotProcess != null)
                .findAny();

            Optional<ShardWithStats> writeShard = Arrays.stream(shards)
                .filter(s -> s.shard.writeProcess != null)
                .collect(CollectorsF.randomOptional());

            Optional<ShardWithStats> writeErrorShard = Arrays.stream(shards)
                .filter(s -> {
                    LogProcess writeProcess = s.shard.writeProcess;
                    return writeProcess != null && writeProcess.lastError != null;
                })
                .collect(CollectorsF.randomOptional());

            Optional<ShardWithStats> readShard = Arrays.stream(shards)
                .filter(s -> s.waitingReadRequests != 0)
                .collect(CollectorsF.randomOptional());

            hw.tableTable(() -> {
                new ShardWriter().writeTrThs();
                new ShardWriter().writeShardRow("Lg by 2h sns", largestBySnapshots);
                new ShardWriter().writeShardRow("Lg by records", largestByRecords);
                new ShardWriter().writeShardRow("Lg by act ratio", largestByActRatio);
                new ShardWriter().writeShardRow("Lg by act/s", byActsPerSecond.max);
                new ShardWriter().writeShardRow("Sm by act/s", byActsPerSecond.min);
                new ShardWriter().writeShardRow("Lg by wr tx/s", byWriteTxsPerSecond.max);
                new ShardWriter().writeShardRow("Sm by wr tx/s", byWriteTxsPerSecond.min);
                new ShardWriter().writeShardRow("Lg by wr recs/s", byWriteRecordsPerSecond.max);
                new ShardWriter().writeShardRow("Sm by wr recs/s", byWriteRecordsPerSecond.min);
                new ShardWriter().writeShardRow("Oldest dl merge", oldestDailyMerge);
                new ShardWriter().writeShardRow("Oldest et merge", oldestEternityMerge);
                new ShardWriter().writeShardRow("Oldest 2h sn", oldestSnapshot);
                new ShardWriter().writeShardRow("Dl merge", dailyMergeShard.orElse(null));
                new ShardWriter().writeShardRow("Et merge", eternityMergeShard.orElse(null));
                new ShardWriter().writeShardRow("Snapshot", snapshotShard.orElse(null));
                new ShardWriter().writeShardRow("Any write", writeShard.orElse(null));
                new ShardWriter().writeShardRow("Any write with err", writeErrorShard.orElse(null));
                new ShardWriter().writeShardRow("Any read", readShard.orElse(null));
            });
        }
    }

    private static <A> Optional<MinMax<A>> minMaxOptional(A[] array, Comparator<A> comparator) {
        Optional<A> min = Arrays.stream(array).min(comparator);
        Optional<A> max = Arrays.stream(array).max(comparator);
        if (min.isPresent() && max.isPresent()) {
            return Optional.of(new MinMax<>(min.get(), max.get()));
        } else if (!min.isPresent() && !max.isPresent()) {
            return Optional.empty();
        } else {
            throw new IllegalStateException();
        }
    }

    private static <A> MinMax<A> minMax(A[] array, Comparator<A> comparator) {
        return minMaxOptional(array, comparator).orElseThrow(() -> new RuntimeException("empty"));
    }

    private static class ShardWithStats {
        private final StockpileShard shard;
        private final int twoHourSnapshots;
        private final long recordCount;
        private final int waitingReadRequests;
        private final float actRatio;
        private final float actPerSecond;
        private final float writeTxsPerSecond;
        private final float writeRecordsPerSecond;
        private final SnapshotTs dailyMergeInstant;
        private final SnapshotTs eternityMergeInstant;
        private final SnapshotTs snapshotInstant;

        public ShardWithStats(StockpileShard shard) {
            this.shard = shard;
            this.twoHourSnapshots = shard.snapshotCount(SnapshotLevel.TWO_HOURS).orElse(-1);
            this.recordCount = shard.indexStats().getTotalByLevel().getTotalByProjects().getTotalByKinds().records;
            this.waitingReadRequests = shard.getWaitingRequestCountForMon();
            this.actRatio = shard.stats.actRatio();
            this.actPerSecond = shard.stats.actPerSecond();
            this.writeTxsPerSecond = shard.stats.writeTxsPerSecond();
            this.writeRecordsPerSecond = shard.stats.writeRecordsPerSecond();
            this.dailyMergeInstant = shard.latestSnapshotTime(SnapshotLevel.DAILY);
            this.eternityMergeInstant = shard.latestSnapshotTime(SnapshotLevel.ETERNITY);
            this.snapshotInstant = shard.latestSnapshotTime(SnapshotLevel.TWO_HOURS);
        }
    }

    private static class MinMax<A> {
        private final A min;
        private final A max;

        public MinMax(A min, A max) {
            this.min = min;
            this.max = max;
        }
    }
}
