package ru.yandex.stockpile.server.shard;

import java.util.List;
import java.util.function.Supplier;

import ru.yandex.monlib.metrics.MetricConsumer;
import ru.yandex.monlib.metrics.MetricSupplier;
import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.meter.Meter;
import ru.yandex.monlib.metrics.primitives.GaugeInt64;
import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.util.TimeSpentInStates;
import ru.yandex.stockpile.server.data.names.FileKind;
import ru.yandex.stockpile.server.shard.actor.StockpileShardActState;
import ru.yandex.stockpile.server.shard.cache.MetricDataCache;
import ru.yandex.stockpile.server.shard.stat.SizeAndCount;
import ru.yandex.stockpile.server.shard.stat.StockpileShardDiskStats;

/**
 * @author Vladimir Gordiychuk
 */
public class StockpileShardMetrics implements MetricSupplier {
    private final String shardId;

    public final ActorMetrics act;
    public final CacheMetrics cache;
    public final WriteMetrics write;
    public final ReadMetrics read;
    public final AllocateIdsMetrics allocateIds;
    public final DiskMetrics disk;
    public final Rate errors;
    final GaugeInt64 records;
    final GaugeInt64 metrics;
    public final Meter utimeNanos;
    final GaugeInt64 memory;
    final GaugeInt64 memoryLimit;

    final MetricRegistry registry;

    StockpileShardMetrics(String shardId) {
        this.shardId = shardId;
        this.registry = new MetricRegistry();
        final String prefix = "stockpile.shard.";

        act = new ActorMetrics(registry, prefix + "act.");
        cache = new CacheMetrics(registry, prefix + "cache.");
        write = new WriteMetrics(registry, prefix + "write.");
        read = new ReadMetrics(registry, prefix + "read.");
        allocateIds = new AllocateIdsMetrics(registry, prefix + "allocateIds.");
        disk = new DiskMetrics(prefix + "disk.");
        errors = registry.rate(prefix + "errors");
        records = registry.gaugeInt64(prefix + "records");
        metrics = registry.gaugeInt64(prefix + "metrics");
        utimeNanos = registry.fiveMinutesMeter(prefix + "utimeNanos");
        memory = registry.gaugeInt64(prefix + "memory");
        memoryLimit = registry.gaugeInt64(prefix + "memory.limit");
    }

    public void measureShardUtime(Runnable runnable) {
        long startNanos = System.nanoTime();
        try {
            runnable.run();
        } finally {
            utimeNanos.mark(System.nanoTime() - startNanos);
        }
    }

    public <T> T measureShardUtime(Supplier<T> supplier) {
        long startNanos = System.nanoTime();
        try {
            return supplier.get();
        } finally {
            utimeNanos.mark(System.nanoTime() - startNanos);
        }
    }

    @Override
    public int estimateCount() {
        return registry.estimateCount();
    }

    @Override
    public void append(long tsMillis, Labels commonLabels, MetricConsumer consumer) {
        commonLabels = commonLabels.toBuilder()
                .add("shardId", shardId)
                .add("host", "")
                .build();

        if ("total".equals(shardId)) {
            commonLabels = commonLabels.removeByKey("host");
        }

        doAppend(tsMillis, commonLabels, consumer);
    }

    private void doAppend(long tsMillis, Labels commonLabels, MetricConsumer consumer) {
        registry.append(tsMillis, commonLabels, consumer);
        act.append(tsMillis, commonLabels, consumer);
        disk.append(tsMillis, commonLabels, consumer);
    }

    public void combine(StockpileShardMetrics target) {
        this.act.combine(target.act);
        this.cache.combine(target.cache);
        this.write.combine(target.write);
        this.read.combine(target.read);
        this.allocateIds.combine(target.allocateIds);
        this.disk.combine(target.disk);
        this.errors.combine(target.errors);
        this.records.combine(target.records);
        this.metrics.combine(target.metrics);
        this.utimeNanos.combine(target.utimeNanos);
        this.memory.combine(target.memory);
        this.memoryLimit.combine(target.memoryLimit);
    }

    public static class CacheMetrics {
        final GaugeInt64 bytesUsed;
        final GaugeInt64 bytesMax;

        final GaugeInt64 metricsCount;

        final Rate bytesAdded;
        final Rate metricsAdded;

        final Rate bytesRemoved;
        final Rate metricsRemoved;

        final Rate hit;
        final Rate miss;
        public final Meter avgMiss;

        CacheMetrics(MetricRegistry registry, String prefix) {
            bytesUsed = registry.gaugeInt64(prefix + "bytes.used");
            bytesMax = registry.gaugeInt64(prefix + "bytes.max");

            metricsCount = registry.gaugeInt64(prefix + "sensors.count");

            bytesAdded = registry.rate(prefix + "bytes.added");
            metricsAdded = registry.rate(prefix + "sensors.added");

            bytesRemoved = registry.rate(prefix + "bytes.removed");
            metricsRemoved = registry.rate(prefix + "sensors.removed");

            hit = registry.rate(prefix + "hit");
            miss = registry.rate(prefix + "miss");
            avgMiss = registry.fiveMinutesMeter(prefix + "avg.miss");
        }

        void update(MetricDataCache cache) {
            this.bytesUsed.set(cache.getBytesUsed());
            this.bytesMax.set(cache.getMaxSizeBytes());
            this.metricsCount.set(cache.getRecordsUsed());
            this.bytesAdded.set(cache.getAddedBytes());
            this.metricsAdded.set(cache.getAddedRecords());
            this.bytesRemoved.set(cache.getRemovedBytes());
            this.metricsRemoved.set(cache.getRemovedRecords());
            this.hit.set(cache.getHit());
            this.miss.set(cache.getMiss());
        }

        private void combine(CacheMetrics target) {
            this.bytesUsed.combine(target.bytesUsed);
            this.bytesMax.combine(target.bytesMax);
            this.metricsCount.combine(target.metricsCount);
            this.bytesAdded.combine(target.bytesAdded);
            this.metricsAdded.combine(target.metricsAdded);
            this.bytesRemoved.combine(target.bytesRemoved);
            this.metricsRemoved.combine(target.metricsRemoved);
            this.hit.combine(target.hit);
            this.miss.combine(target.miss);
            this.avgMiss.combine(target.avgMiss);
        }
    }

    public static class ActorMetrics {
        public final Rate time;
        public final Rate count;
        final TimeSpentInStates<StockpileShardActState> timeByState;

        ActorMetrics(MetricRegistry registry, String prefix) {
            timeByState = new TimeSpentInStates<>(StockpileShardActState.class, prefix + "state.times");
            time = registry.rate(prefix + "times");
            count = registry.rate(prefix + "count");
        }

        void switchToState(StockpileShardActState state) {
            timeByState.switchToState(state);
        }

        void combine(ActorMetrics target) {
            this.timeByState.combine(target.timeByState);
            this.time.combine(target.time);
            this.count.combine(target.count);
        }

        void append(long tsMillis, Labels commonLabels, MetricConsumer consumer) {
            timeByState.append(tsMillis, commonLabels, consumer);
        }
    }

    public static class WriteMetrics {
        public final Rate txs;
        public final Rate requests;
        public final Rate metrics;
        public final Rate records;
        public final Rate bytes;
        public final Rate ignoreRequests;
        public final Rate ignoreMetrics;
        public final Rate ignoreRecords;
        public final Rate ignoreBytes;

        public final GaugeInt64 queueBytes;
        public final GaugeInt64 queueRequests;
        public final GaugeInt64 txInFlight;

        public final Meter avgRecordsRps;
        public final Meter avgMetricsRps;
        public final Meter avgLogWriteMillis;
        public final Meter avgLogSnapshotWriteMillis;

        public WriteMetrics(MetricRegistry registry, String prefix) {
            txs = registry.rate(prefix + "txs");
            requests = registry.rate(prefix + "requests");
            metrics = registry.rate(prefix + "sensors");
            records = registry.rate(prefix + "records");
            bytes = registry.rate(prefix + "bytes");
            ignoreRequests = registry.rate(prefix + "ignoreRequests");
            ignoreMetrics = registry.rate(prefix + "ignoreSensors");
            ignoreRecords = registry.rate(prefix + "ignoreRecords");
            ignoreBytes = registry.rate(prefix + "ignoreBytes");

            queueBytes = registry.gaugeInt64(prefix + "queue.bytes");
            queueRequests = registry.gaugeInt64(prefix + "queue.requests");
            avgLogWriteMillis = registry.fiveMinutesMeter(prefix + "log.avg.elapsedTimeMillis", Labels.of("kind", "short"));
            avgLogSnapshotWriteMillis = registry.fiveMinutesMeter(prefix + "log.avg.elapsedTimeMillis", Labels.of("kind", "snapshot"));
            avgRecordsRps = registry.fiveMinutesMeter(prefix + "avg.records.rate");
            avgMetricsRps = registry.fiveMinutesMeter(prefix + "avg.sensors.rate");
            txInFlight = registry.gaugeInt64(prefix + "tx.inFlight");
        }

        void update(TxWriteSummary summary) {
            txs.inc();
            requests.add(summary.requests);
            metrics.add(summary.metrics);
            records.add(summary.records);
            bytes.add(summary.bytes);

            avgRecordsRps.mark(summary.records);
            avgMetricsRps.mark(summary.metrics);
        }

        void ignore(List<StockpileWriteRequest> requests) {
            int metrics = 0;
            int records = 0;
            int bytes = 0;
            for (var req : requests) {
                var archives = req.getArchiveByLocalId();
                if (archives == null) {
                    continue;
                }

                metrics += archives.size();
                records += req.getRecordsInArchives();
                bytes += req.getMemorySizeInArchives();
            }

            ignoreRequests.add(requests.size());
            ignoreMetrics.add(metrics);
            ignoreRecords.add(records);
            ignoreBytes.add(bytes);
        }

        private void combine(WriteMetrics target) {
            this.txs.combine(target.txs);
            this.requests.combine(target.requests);
            this.metrics.combine(target.metrics);
            this.records.combine(target.records);
            this.bytes.combine(target.bytes);
            this.ignoreRequests.combine(target.ignoreRequests);
            this.ignoreMetrics.combine(target.ignoreMetrics);
            this.ignoreRecords.combine(target.ignoreRecords);
            this.ignoreBytes.combine(target.ignoreBytes);

            this.queueBytes.combine(target.queueBytes);
            this.queueRequests.combine(target.queueRequests);

            this.avgRecordsRps.combine(target.avgRecordsRps);
            this.avgMetricsRps.combine(target.avgMetricsRps);
            this.avgLogWriteMillis.combine(target.avgLogWriteMillis);
            this.avgLogSnapshotWriteMillis.combine(target.avgLogSnapshotWriteMillis);
        }
    }

    public static class ReadMetrics {
        public final GaugeInt64 queueBytes;
        public final GaugeInt64 queueRequests;

        public final Meter avgRecordsRps;
        public final Meter avgMetricsRps;

        public ReadMetrics(MetricRegistry registry, String prefix) {
            queueBytes = registry.gaugeInt64(prefix + "queue.bytes");
            queueRequests = registry.gaugeInt64(prefix + "queue.requests");
            avgRecordsRps = registry.fiveMinutesMeter(prefix + "avg.records.rate");
            avgMetricsRps = registry.fiveMinutesMeter(prefix + "avg.sensors.rate");
        }

        public void combine(ReadMetrics read) {
            this.queueBytes.combine(read.queueBytes);
            this.queueRequests.combine(read.queueRequests);
            this.avgRecordsRps.combine(read.avgRecordsRps);
            this.avgMetricsRps.combine(read.avgMetricsRps);
        }
    }

    public static class AllocateIdsMetrics {
        public final GaugeInt64 queueBytes;
        public final GaugeInt64 queueRequests;

        public AllocateIdsMetrics(MetricRegistry registry, String prefix) {
            queueBytes = registry.gaugeInt64(prefix + "queue.bytes");
            queueRequests = registry.gaugeInt64(prefix + "queue.requests");
        }

        public void combine(AllocateIdsMetrics read) {
            this.queueBytes.combine(read.queueBytes);
            this.queueRequests.combine(read.queueRequests);
        }
    }

    public static class DiskMetrics implements MetricSupplier {
        private final String prefix;
        volatile StockpileShardDiskStats stats = new StockpileShardDiskStats();

        public DiskMetrics(String prefix) {
            this.prefix = prefix;
        }

        @Override
        public int estimateCount() {
            return FileKind.values().length * 2;
        }

        @Override
        public void append(long tsMillis, Labels commonLabels, MetricConsumer consumer) {
            for (FileKind kind : FileKind.values()) {
                SizeAndCount size = stats.get(kind);

                // stockpile.shard.disk.file.count{type=...}
                consumer.onMetricBegin(MetricType.IGAUGE);
                consumer.onLabelsBegin(commonLabels.size() + 2);
                commonLabels.forEach(consumer::onLabel);
                consumer.onLabel("sensor", prefix + "file.count");
                consumer.onLabel("type", kind.monKey);
                consumer.onLabelsEnd();
                consumer.onLong(tsMillis, size.count());
                consumer.onMetricEnd();

                // stockpile.shard.disk.file.bytes{type=...}
                consumer.onMetricBegin(MetricType.IGAUGE);
                consumer.onLabelsBegin(commonLabels.size() + 2);
                commonLabels.forEach(consumer::onLabel);
                consumer.onLabel("sensor", prefix + "file.bytes");
                consumer.onLabel("type", kind.monKey);
                consumer.onLabelsEnd();
                consumer.onLong(tsMillis, size.size());
                consumer.onMetricEnd();
            }
        }

        void update(StockpileShardDiskStats update) {
            this.stats = update;
        }

        public void combine(DiskMetrics target) {
            stats = StockpileShardDiskStats.add(stats, target.stats);
        }
    }
}
