package ru.yandex.stockpile.server.shard;

import java.util.EnumMap;
import java.util.Optional;
import java.util.OptionalInt;
import java.util.OptionalLong;
import java.util.stream.Stream;

import javax.annotation.Nonnull;

import ru.yandex.misc.lang.EnumMapUtils;
import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.memory.layout.MemoryBySubsystem;
import ru.yandex.stockpile.server.SnapshotLevel;
import ru.yandex.stockpile.server.data.chunk.IndexRangeResult;
import ru.yandex.stockpile.server.data.index.SnapshotIndex;
import ru.yandex.stockpile.server.data.index.stats.IndexStatsLevel;
import ru.yandex.stockpile.server.shard.actor.InActor;
import ru.yandex.stockpile.server.shard.stat.LevelSizeAndCount;
import ru.yandex.stockpile.server.shard.stat.StockpileShardDiskStats;

/**
 * @author Stepan Koltsov
 */
public class StockpileShardStateDone extends StockpileShardState {

    @Nonnull
    final AllIndexes indexes;

    public StockpileShardStateDone(StockpileShard shard, @Nonnull AllIndexes indexes) {
        super(shard);
        this.indexes = indexes;
    }


    @Override
    protected Stream<SnapshotIndexWithStats> indexesWithStatsUnsafe() {
        return indexes.streamWithStats();
    }

    Stream<SnapshotIndex> indexes(InActor a) {
        return indexesUnsafe();
    }

    @Override
    public StockpileShard.LoadState loadStateForMon() {
        return StockpileShard.LoadState.DONE;
    }

    @Override
    public StockpileShardDiskStats diskStats() {
        EnumMap<SnapshotLevel, LevelSizeAndCount> levelDiskStats =
            EnumMapUtils.fill(SnapshotLevel.class, level -> indexes.diskSize(level));

        return new StockpileShardDiskStats(
            shard.logState.diskStats(),
            levelDiskStats);

    }

    @Override
    public OptionalInt snapshotCount(SnapshotLevel level) {
        return OptionalInt.of(indexes.snapshotCount(level));
    }

    public IndexRangeResult rangeRequestsForMetric(InActor a, long localId, long fromMillis) {
        return indexes.rangeRequestsForMetric(localId, fromMillis);
    }

    @Override
    public SnapshotTs latestStapshotTime(SnapshotLevel snapshotLevel) {
        long tsMillis = indexes.latestSnapshotTime(snapshotLevel);
        if (tsMillis == 0) {
            tsMillis = SnapshotTs.SpecialTs.NEVER.value;
        }
        return new SnapshotTs(tsMillis);
    }

    @Override
    public IndexStatsLevel indexesStats() {
        return indexes.getStats();
    }

    @Override
    public long minRequiredMemory() {
        return indexes.memorySizeIncludingSelf();
    }

    @Override
    public OptionalLong recordCountReliable() {
        return OptionalLong.of(indexesStats().getTotalByLevel().getTotalByProjects().getTotalByKinds().records);
    }

    @Override
    public OptionalLong metricsCountReliable() {
        return OptionalLong.of(indexesStats().getTotalByLevel().getTotalByProjects().getTotalByKinds().metrics);
    }

    @Override
    public Optional<StockpileFormat> oldestUsedFormatForMon() {
        return indexes.stream()
            .filter(SnapshotIndex::isReal)
            .map(i -> i.getContent().getFormat())
            .min(StockpileFormat::compare);
    }

    @Override
    public void stop(InActor a) {
        indexes.destroy();
    }

    @Override
    public void addMemoryBySubsystem(MemoryBySubsystem memory) {
        memory.addAllMemory(indexes.memoryBySystem());
    }
}
