package ru.yandex.stockpile.server.shard;

import java.util.OptionalInt;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Stream;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.solomon.memory.layout.MemoryBySubsystem;
import ru.yandex.stockpile.client.shard.StockpileShardId;
import ru.yandex.stockpile.kikimrKv.counting.ReadClass;
import ru.yandex.stockpile.server.SnapshotLevel;
import ru.yandex.stockpile.server.shard.stat.StockpileShardDiskStats;

/**
 * @author Stepan Koltsov
 */
public class StockpileShardStateInit extends StockpileShardState {
    private static final Logger logger = LoggerFactory.getLogger(StockpileShardStateInit.class);
    private volatile boolean locked;

    public StockpileShardStateInit(StockpileShard shard) {
        super(shard);
    }

    @Override
    protected Stream<SnapshotIndexWithStats> indexesWithStatsUnsafe() {
        return Stream.empty();
    }

    @Override
    public StockpileShard.LoadState loadStateForMon() {
        return locked ? StockpileShard.LoadState.LOCKED : StockpileShard.LoadState.INIT;
    }

    @Override
    public StockpileShardDiskStats diskStats() {
        return StockpileShardDiskStats.zeroes();
    }

    @Override
    public OptionalInt snapshotCount(SnapshotLevel level) {
        return OptionalInt.empty();
    }

    protected CompletableFuture<Boolean> canLoadData() {
        return loopUntilSuccessFuture("checkLockFile", () -> {
            return shard.globals.kikimrKvClient.readData(ReadClass.OTHER, shard.kvTabletId, 0, KvLock.FILE)
                .thenApply(result -> {
                    if (result.isEmpty()) {
                        logger.info("Shard {} .lock file absent", StockpileShardId.toString(shard.shardId));
                        return Boolean.TRUE;
                    }

                    var bytes = result.get();
                    if (bytes.length == 0) {
                        logger.info("Shard {} .lock file empty", StockpileShardId.toString(shard.shardId));
                        return Boolean.TRUE;
                    }

                    KvLock lock;
                    try {
                        lock = KvLock.parse(bytes);
                    } catch (Throwable e) {
                        logger.error("Shard {} .lock file, parse failed", StockpileShardId.toString(shard.shardId), e);
                        return Boolean.TRUE;
                    }

                    if (lock.isExpired()) {
                        logger.info("Shard {} .lock file expired {}", StockpileShardId.toString(shard.shardId), lock);
                        return Boolean.TRUE;
                    }

                    locked = true;
                    logger.warn("Shard {} init locked by {}", StockpileShardId.toString(shard.shardId), lock);
                    return Boolean.FALSE;
                });
        });
    }

    public boolean isLocked() {
        return locked;
    }

    @Override
    public void addMemoryBySubsystem(MemoryBySubsystem memory) {
    }
}
