package ru.yandex.stockpile.server.shard;

import java.util.concurrent.CompletableFuture;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import it.unimi.dsi.fastutil.longs.Long2ObjectOpenHashMap;

import ru.yandex.solomon.codec.archive.MetricArchiveImmutable;
import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.memory.layout.MemoryCounter;
import ru.yandex.stockpile.server.Txn;
import ru.yandex.stockpile.server.data.DeletedShardSet;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class StockpileWriteRequest implements StockpileRequest {
    private static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(StockpileWriteRequest.class);
    @Nullable
    private Long2ObjectOpenHashMap<MetricArchiveMutable> archiveByLocalId;
    @Nullable
    private DeletedShardSet deletedShards;
    private final int producerId;
    private final long producerSeqNo;
    private final long memorySizeInArchives;
    private final int recordsInArchive;
    private final CompletableFuture<Txn> future;
    private final long createdAtNanos;

    public StockpileWriteRequest(Long2ObjectOpenHashMap<MetricArchiveMutable> archiveByLocalId, int producerId, long producerSeqNo) {
        this.archiveByLocalId = archiveByLocalId;
        this.producerId = producerId;
        this.producerSeqNo = producerSeqNo;
        this.future = new CompletableFuture<>();
        this.createdAtNanos = System.nanoTime();

        long bytes = 0;
        int records = 0;
        for (MetricArchiveMutable archive : archiveByLocalId.values()) {
            bytes += archive.memorySizeIncludingSelf();
            records += archive.getRecordCount() + 1; // header it's also record
        }

        this.memorySizeInArchives = bytes;
        this.recordsInArchive = records;
    }

    public StockpileWriteRequest(Long2ObjectOpenHashMap<MetricArchiveMutable> archiveByLocalId, @Nullable DeletedShardSet deletedShards, int producerId, long producerSeqNo) {
        this.archiveByLocalId = archiveByLocalId;
        this.producerId = producerId;
        this.producerSeqNo = producerSeqNo;
        this.deletedShards = deletedShards;
        this.future = new CompletableFuture<>();
        this.createdAtNanos = System.nanoTime();

        long bytes = 0;
        int records = 0;
        for (MetricArchiveMutable archive : archiveByLocalId.values()) {
            bytes += archive.memorySizeIncludingSelf();
            records += archive.getRecordCount() + 1; // header it's also record
        }

        this.memorySizeInArchives = bytes;
        this.recordsInArchive = records;
    }

    public CompletableFuture<Txn> getFuture() {
        return future;
    }

    public int getProducerId() {
        return producerId;
    }

    public long getProducerSeqNo() {
        return producerSeqNo;
    }

    @Override
    public long getCreatedAtNanos() {
        return createdAtNanos;
    }

    public long getMemorySizeInArchives() {
        return memorySizeInArchives;
    }

    public int getRecordsInArchives() {
        return recordsInArchive;
    }

    public boolean isEmpty() {
        if (recordsInArchive != 0) {
            return false;
        }
        if (deletedShards == null) {
            return true;
        }
        return deletedShards.isEmpty();
    }

    @Nullable
    public Long2ObjectOpenHashMap<MetricArchiveMutable> getArchiveByLocalId() {
        return archiveByLocalId;
    }

    @Nullable
    public DeletedShardSet getDeletedShardSet() {
        return deletedShards;
    }

    public void release() {
        var copy = archiveByLocalId;
        if (copy != null) {
            for (var archive : copy.values()) {
                archive.close();
            }
            copy.clear();
            archiveByLocalId = null;
        }
        deletedShards = null;
    }

    public void clear() {
        archiveByLocalId = null;
        deletedShards = null;
    }

    @Override
    public long memorySizeIncludingSelf() {
        long size = SELF_SIZE;
        size += MemoryCounter.CompletableFuture_SELF_SIZE;

        Long2ObjectOpenHashMap copy = archiveByLocalId;
        if (copy != null) {
            size += memorySizeInArchives;
            size += MemoryCounter.long2ObjectOpenHashMapSize(copy) + 16;
        }
        return size;
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public static class Builder {
        private final Long2ObjectOpenHashMap<MetricArchiveMutable> archiveByLocalId = new Long2ObjectOpenHashMap<>();
        @Nullable
        private DeletedShardSet deletedShards;
        private int producerId;
        private long producerSeqNo;

        private Builder() {
        }

        public MetricArchiveMutable archiveRef(long localId) {
            var archive = archiveByLocalId.get(localId);
            if (archive == null) {
                archive = new MetricArchiveMutable();
                archiveByLocalId.put(localId, archive);
            }
            return archive;
        }

        public Builder addArchive(long localId, MetricArchiveImmutable archive) {
            var prev = archiveByLocalId.get(localId);
            if (prev == null) {
                archiveByLocalId.put(localId, archive.toMutable());
            } else {
                prev.updateWith(archive);
            }
            return this;
        }

        public Builder addArchive(long localId, MetricArchiveMutable archive) {
            var prev = archiveByLocalId.get(localId);
            if (prev == null) {
                archiveByLocalId.put(localId, archive);
            } else {
                prev.updateWith(archive);
            }
            return this;
        }

        public Builder addArchiveCopy(long localId, MetricArchiveMutable archive) {
            var prev = archiveByLocalId.get(localId);
            if (prev == null) {
                archiveByLocalId.put(localId, archive.cloneToUnsealed());
            } else {
                prev.updateWith(archive);
            }
            return this;
        }

        public Builder setProducerId(int producerId) {
            this.producerId = producerId;
            return this;
        }

        public Builder setProducerSeqNo(long producerSeqNo) {
            this.producerSeqNo = producerSeqNo;
            return this;
        }

        public Builder addDeleteShard(int projectId, int shardId) {
            if (deletedShards == null) {
                deletedShards = new DeletedShardSet();
            }
            deletedShards.add(projectId, shardId);
            return this;
        }

        public StockpileWriteRequest build() {
            return new StockpileWriteRequest(archiveByLocalId, deletedShards, producerId, producerSeqNo);
        }
    }
}
