package ru.yandex.stockpile.server.shard;

import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Stream;

import javax.annotation.Nonnull;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.memory.layout.MemoryCounter;
import ru.yandex.solomon.util.ExceptionUtils;
import ru.yandex.stockpile.memState.LogEntriesContent;
import ru.yandex.stockpile.server.SnapshotLevel;
import ru.yandex.stockpile.server.shard.actor.ActorRunnable;
import ru.yandex.stockpile.server.shard.actor.ActorRunnableType;
import ru.yandex.stockpile.server.shard.actor.InActor;
import ru.yandex.stockpile.server.shard.actor.StockpileShardActState;

/**
 * @author Stepan Koltsov
 */
class TwoHourSnapshotProcess extends ShardProcess {
    public static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(TwoHourSnapshotProcess.class);
    private static final Logger logger = LoggerFactory.getLogger(TwoHourSnapshotProcess.class);

    @Nonnull
    private final StockpileShardStateDone shardStateDone;
    private final List<CompletableFuture<Long>> snapshotCompletedFutures;

    private final long snapshotTxn;
    private final SnapshotReason snapshotReason;

    // used to serve reads
    LogEntriesContent currentlyWrittenSnapshot = new LogEntriesContent();

    public TwoHourSnapshotProcess(@Nonnull StockpileShardStateDone shardStateDone, List<CompletableFuture<Long>> snapshotCompletedFutures, SnapshotReason snapshotReason, long txn, InActor a) {
        super(shardStateDone.shard, ProcessType.TWO_HOUR_SNAPSHOT, "reason: " + snapshotReason + ", txn " + txn);
        this.shardStateDone = shardStateDone;

        if (shard.twoHourSnapshotProcess != null) {
            throw new IllegalStateException();
        }

        this.snapshotCompletedFutures = snapshotCompletedFutures;
        this.snapshotReason = snapshotReason;
        this.snapshotTxn = txn;
    }

    public Stream<MetricArchiveMutable> getMemoryState(long localId) {
        var archive = currentlyWrittenSnapshot.getMetricToArchiveMap().getById(localId);
        return Stream.ofNullable(archive);
    }

    @Override
    public void start(InActor a) {
        if (shard.twoHourSnapshotProcess != this) {
            throw new IllegalStateException("Multiple two hour snapshot process not allowed");
        }

        // await until finish inflight writes, before start two hours snapshot, otherwise,
        // transactions can be lost after restart:
        // 1 [commit]
        // 2 [commit]
        // 3   [in-flight]
        // 4 [snapshot]
        // after restart it will looks like
        // 4. [snapshot without 3]
        shard.txTracker.written(new TxTracker.Tx() {
            @Override
            public long txn() {
                return snapshotTxn;
            }

            @Override
            public void completeTx() {
                shard.metrics.act.switchToState(StockpileShardActState.TWO_HOUR_SNAPSHOT);
                currentlyWrittenSnapshot = shard.logState.takeForTwoHourSnapshot();
                shard.commonExecutor.execute(() -> {
                    if (shard.isGenerationChanged() || shard.isStop()) {
                        return;
                    }

                    if (currentlyWrittenSnapshot.isEmpty()) {
                        ExceptionUtils.uncaughtException(new IllegalStateException("not able make two hours snapshot without metrics: " + snapshotTxn));
                    }

                    runInner();
                });
            }
        });
    }

    private void runInner() {
        Write2hSnapshotDeleteLogs write2hSnapshotDeleteLogs = new Write2hSnapshotDeleteLogs(this, snapshotTxn, snapshotReason, currentlyWrittenSnapshot);
        write2hSnapshotDeleteLogs.run()
            .thenAccept(dataWritten -> {
                processRunInActor(new ActorRunnable() {
                    @Override
                    public ActorRunnableType type() {
                        return ActorRunnableType.MISC;
                    }

                    @Override
                    public void run(InActor a) {
                        renamed(a, dataWritten);
                    }
                });
            })
            .whenComplete((unit, throwable) -> {
                if (throwable != null && !ExceptionHandler.isGenerationChanged(throwable)) {
                    RuntimeException e = new RuntimeException("uncaught exception in shard: " + shard.shardId, throwable);
                    ExceptionUtils.uncaughtException(e);
                }
            });
    }

    private void renamed(InActor a, SnapshotIndexWithStats snapshot) {
        if (shard.twoHourSnapshotProcess != this) {
            throw new IllegalStateException();
        }

        releaseSnapshot();
        shardStateDone.indexes.addSnapshot(snapshot);
        shardStateDone.indexes.updateLatestSnapshotTime(SnapshotLevel.TWO_HOURS, snapshot.getIndex().getContent().getTsMillis());

        shard.snapshotPeriodMillis = SnapshotAndMergeScheduler.randomSnapshotPeriod();
        for (CompletableFuture<Long> future : snapshotCompletedFutures) {
            future.complete(snapshotTxn);
        }

        shard.twoHourSnapshotProcess = null;
        shard.actorRunner.schedule();

        completedSuccessfullyWriteStats();
    }

    private void releaseSnapshot() {
        var copy = currentlyWrittenSnapshot;
        currentlyWrittenSnapshot = new LogEntriesContent();
        copy.release();
    }

    @Override
    protected void stoppedReleaseResources() {
        RuntimeException e = shard.stopException();
        for (CompletableFuture<Long> future : snapshotCompletedFutures) {
            future.completeExceptionally(e);
        }
    }

    @Override
    public long memorySizeIncludingSelf() {
        var copy = currentlyWrittenSnapshot;
        return SELF_SIZE + copy.memorySizeIncludingSelf();
    }
}
