package ru.yandex.stockpile.server.shard;

import java.util.List;
import java.util.Objects;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.PooledByteBufAllocator;
import it.unimi.dsi.fastutil.ints.Int2LongMap;
import it.unimi.dsi.fastutil.ints.Int2LongOpenHashMap;
import it.unimi.dsi.fastutil.longs.Long2ObjectOpenHashMap;

import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.codec.serializer.NettyStockpileSerializer;
import ru.yandex.solomon.memory.layout.MemMeasurable;
import ru.yandex.solomon.memory.layout.MemoryCounter;
import ru.yandex.stockpile.server.Txn;
import ru.yandex.stockpile.server.data.DeletedShardSet;
import ru.yandex.stockpile.server.data.log.StockpileLogEntryContent;
import ru.yandex.stockpile.server.data.log.StockpileLogEntryContentSerializer;
import ru.yandex.stockpile.server.shard.stat.TxWriteStats;

/**
 * @author Vladimir Gordiychuk
 */
public class TxLogEntry implements MemMeasurable {
    private static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(TxLogEntry.class);
    private final Long2ObjectOpenHashMap<MetricArchiveMutable> archiveByLocalId;
    private final Int2LongMap producerSeqNoById = new Int2LongOpenHashMap();
    private final DeletedShardSet deletedShardSet;
    private final TxWriteStats stats;
    private long memorySizeInArchives;
    private int recordsInArchives;
    private int requests;
    private int bytes;

    public TxLogEntry(StockpileWriteRequest request) {
        this.archiveByLocalId = Objects.requireNonNull(request.getArchiveByLocalId());
        this.deletedShardSet = request.getDeletedShardSet() != null ? request.getDeletedShardSet() : new DeletedShardSet();
        this.stats = TxWriteStats.of(archiveByLocalId.values());
        this.memorySizeInArchives = request.getMemorySizeInArchives();
        this.recordsInArchives = request.getRecordsInArchives();
        this.producerSeqNoById.put(request.getProducerId(), request.getProducerSeqNo());
        this.requests = 1;
        request.clear();
    }

    public static TxLogEntry of(List<StockpileWriteRequest> requests) {
        TxLogEntry logEntry = new TxLogEntry(requests.get(0));
        for (int index = 1; index < requests.size(); index++) {
            logEntry.append(requests.get(index));
        }
        logEntry.sortAndMerge();
        return logEntry;
    }

    private void append(StockpileWriteRequest request) {
        try {
            if (request.getDeletedShardSet() != null) {
                deletedShardSet.addAll(request.getDeletedShardSet());
            }

            if (request.getArchiveByLocalId() == null) {
                return;
            }

            int producerId = request.getProducerId();
            long producerSeqNo = request.getProducerSeqNo();
            if (producerId != 0) {
                if (producerSeqNoById.get(producerId) >= producerSeqNo) {
                    request.getFuture().completeAsync(() -> new Txn(1));
                    return;
                }
                producerSeqNoById.put(producerId, producerSeqNo);
            }
            requests++;
            var it = request.getArchiveByLocalId().long2ObjectEntrySet().fastIterator();
            while (it.hasNext()) {
                var entry = it.next();
                long localId = entry.getLongKey();
                var delta = entry.getValue();

                var archive = archiveByLocalId.get(localId);
                if (archive == null) {
                    archiveByLocalId.put(localId, delta);
                    stats.add(delta);
                } else {
                    archive.updateWithNoSortMerge(delta);
                    stats.add(delta);
                    delta.close();
                }
                it.remove();
            }
            recordsInArchives += request.getRecordsInArchives();
            memorySizeInArchives += request.getMemorySizeInArchives();
            request.clear();
        } catch (Throwable e) {
            request.release();
            request.getFuture().completeExceptionally(e);
        }
    }

    public void sortAndMerge() {
        archiveByLocalId
            .values()
            .parallelStream()
            .forEach(MetricArchiveMutable::sortAndMerge);
    }

    public long estimateSerializedSize() {
        if (bytes != 0) {
            return bytes;
        }

        return memorySizeInArchives + archiveByLocalId.size() * Long.BYTES;
    }

    public TxWriteSummary getSummary() {
        return new TxWriteSummary(requests, archiveByLocalId.size(), recordsInArchives, estimateSerializedSize());
    }

    public TxWriteStats getStats() {
        return stats;
    }

    public int getRecords() {
        return recordsInArchives;
    }

    @Override
    public long memorySizeIncludingSelf() {
        return SELF_SIZE + memorySizeInArchives + MemoryCounter.long2ObjectOpenHashMapSize(archiveByLocalId);
    }

    ByteBuf serialize() {
        var buffer = PooledByteBufAllocator.DEFAULT.buffer(Math.toIntExact(estimateSerializedSize()));
        try {
            var serializer = new NettyStockpileSerializer(buffer);
            StockpileLogEntryContentSerializer.S.serializeToEof(getLogEntryContent(), serializer);
            bytes = buffer.readableBytes();
            return buffer;
        } catch (Throwable e) {
            buffer.release();
            throw new RuntimeException(e);
        }
    }

    StockpileLogEntryContent getLogEntryContent() {
        return new StockpileLogEntryContent(archiveByLocalId, deletedShardSet, producerSeqNoById);
    }

    public void release() {
        for (var archive : archiveByLocalId.values()) {
            archive.close();
        }
        archiveByLocalId.clear();
    }
}
