package ru.yandex.stockpile.server.shard;

import java.util.Comparator;
import java.util.PriorityQueue;

import ru.yandex.solomon.util.ExceptionUtils;

/**
 * @author Stepan Koltsov
 */
public class TxTracker {

    public interface Tx {
        long txn();
        void completeTx();
    }

    private long usedTxn;
    private long completedTxn;

    public TxTracker(long usedTxn) {
        this.usedTxn = usedTxn;
        completedTxn = this.usedTxn;
    }

    public long allocateCompletedTxn() {
        long txn = allocateTx();
        written(new TxNoop(txn));
        return txn;
    }

    // TODO: could be more efficient structure
    private PriorityQueue<Tx> outOfOrderTxs = new PriorityQueue<>(Comparator.comparing(Tx::txn));

    public long allocateTx() {
        return ++usedTxn;
    }

    private void complete(Tx tx) {
        try {
            tx.completeTx();
        } catch (Throwable x) {
            ExceptionUtils.uncaughtException(x);
        }
    }

    private void flushOutOfOrderTxs() {
        for (;;) {
            Tx peek = outOfOrderTxs.peek();

            if (peek == null) {
                break;
            }

            if (peek.txn() != completedTxn + 1) {
                break;
            }

            complete(peek);
            ++completedTxn;
            outOfOrderTxs.poll();
        }
    }

    public void written(Tx tx) {
        if (tx.txn() == completedTxn + 1) {
            complete(tx);
            ++completedTxn;

            flushOutOfOrderTxs();
        } else {
            outOfOrderTxs.add(tx);
        }
    }

    @Override
    public String toString() {
        return "TxTracker{" +
            "usedTxn=" + usedTxn +
            ", completedTxn=" + completedTxn +
            ", outOfOrderTxs=" + outOfOrderTxs +
            '}';
    }
}
