package ru.yandex.stockpile.server.shard;

import java.util.concurrent.CompletableFuture;

import javax.annotation.CheckReturnValue;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.util.actors.AsyncActorBody;
import ru.yandex.solomon.util.actors.AsyncActorRunner;
import ru.yandex.stockpile.memState.LogEntriesContent;
import ru.yandex.stockpile.server.SnapshotLevel;
import ru.yandex.stockpile.server.data.chunk.ChunkWithNo;
import ru.yandex.stockpile.server.data.chunk.SnapshotAddress;
import ru.yandex.stockpile.server.data.command.SnapshotCommandContent;
import ru.yandex.stockpile.server.data.index.SnapshotIndex;
import ru.yandex.stockpile.server.data.index.SnapshotIndexContent;
import ru.yandex.stockpile.server.data.log.StockpileProducerSeqNoSnapshotSerializer;
import ru.yandex.stockpile.server.shard.stat.LevelSizeAndCount;
import ru.yandex.stockpile.server.shard.stat.SizeAndCount;

import static java.util.concurrent.CompletableFuture.completedFuture;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class Write2hSnapshotDeleteLogs {
    private final ShardProcess process;
    private final StockpileShard shard;
    private final long txn;
    private final SnapshotReason reason;
    private final LogEntriesContent content;
    private SizeAndCount commandSize = SizeAndCount.zero;

    public Write2hSnapshotDeleteLogs(ShardProcess process, long snapshotTxn, SnapshotReason reason, LogEntriesContent content) {
        this.process = process;
        this.shard = process.shard;
        this.txn = snapshotTxn;
        this.reason = reason;
        this.content = content;
    }

    @CheckReturnValue
    public CompletableFuture<SnapshotIndexWithStats> run() {
        return writeProducerSeqNoSnapshot()
            .thenCompose(ignore -> writeSnapshotTmp())
            .thenCompose(result -> {
                return renameIndexAndRenameChunksAndDeleteLogs(result.getIndex().snapshotAddress())
                    .thenApply(unit2 -> result);
            });
    }

    private CompletableFuture<SnapshotIndexWithStats> writeSnapshotTmp() {
        return writeCommandsTmp()
                .thenCompose(ignore -> writeChunksTmp())
                .thenCompose(this::writeIndexTmp);
    }

    private CompletableFuture<Void> writeProducerSeqNoSnapshot() {
        if (!StockpileFormat.CURRENT.ge(StockpileFormat.IDEMPOTENT_WRITE_38)) {
            return completedFuture(null);
        }

        byte[] serialized = StockpileProducerSeqNoSnapshotSerializer.S.serializeToBytes(content.getProducerSeqNoById());
        return process.loopUntilSuccessFuture("writeProducerSeqNoSnapshot(" + DataSize.shortString(serialized.length) + ")", () -> {
            return shard.storage.writeProducerSeqNoSnapshot(serialized);
        });
    }

    private CompletableFuture<Void> writeCommandsTmp() {
        var commandContent = new SnapshotCommandContent(content.getDeletedShards());
        if (!SnapshotCommandWriter.isWritable(false, commandContent)) {
            return completedFuture(null);
        }

        var writer = new SnapshotCommandWriter(process);
        return writer.write(SnapshotLevel.TWO_HOURS, txn, commandContent)
                .thenAccept(sizeAndCount -> commandSize = sizeAndCount);
    }

    @CheckReturnValue
    private CompletableFuture<Void> renameIndexAndRenameChunksAndDeleteLogs(SnapshotAddress address) {
        return process.loopUntilSuccessFuture("renameSnapshotDeleteLogs(" + address + ")", () -> {
                return shard.storage.renameSnapshotDeleteLogs(address);
            });
    }

    @CheckReturnValue
    private CompletableFuture<SnapshotIndexContent> writeChunksTmp() {
        LazySnapshotBuilder builder = new LazySnapshotBuilder(reason, content.getMetricToArchiveMap().getMetrics());
        AsyncActorBody body = () -> {
            var chunk = builder.nextChunk();
            if (chunk == null) {
                return completedFuture(AsyncActorBody.DONE_MARKER);
            }
            return writeChunkToTmp(chunk);
        };

        return new AsyncActorRunner(body, shard.commonExecutor, 1)
            .start()
            .thenApply(ignore -> builder.index());
    }

    private CompletableFuture<Void> writeChunkToTmp(ChunkWithNo chunkWithNo) {
        return process.loopUntilSuccessFuture("writeSnapshotChunkTmp(" + chunkWithNo + ")", () -> {
            return shard.storage.writeSnapshotChunkToTemp(SnapshotLevel.TWO_HOURS, txn, chunkWithNo);
        });
    }

    @CheckReturnValue
    private CompletableFuture<SnapshotIndexWithStats> writeIndexTmp(SnapshotIndexContent content) {
        SnapshotIndexWriter writer = new SnapshotIndexWriter(process);
        return writer.write(SnapshotLevel.TWO_HOURS, txn, content)
                .thenApply(indexSize -> {
                    var chunkSize = content.diskSize();
                    var levelSize = new LevelSizeAndCount(indexSize, chunkSize, commandSize);
                    var index = new SnapshotIndex(SnapshotLevel.TWO_HOURS, txn, content);
                    return new SnapshotIndexWithStats(index, levelSize);
                });
    }
}
