package ru.yandex.stockpile.server.shard.cache;

import java.util.List;
import java.util.concurrent.TimeUnit;

import ru.yandex.solomon.codec.BinaryAggrGraphDataListIterator;
import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.codec.archive.header.MetricHeader;
import ru.yandex.solomon.codec.bits.BitBuf;
import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.memory.layout.MemoryCounter;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.solomon.model.timeseries.decim.DecimPoliciesPredefined;
import ru.yandex.solomon.model.timeseries.decim.DecimPolicy;
import ru.yandex.stockpile.server.shard.merge.ClosedArchiveIterable;
import ru.yandex.stockpile.server.shard.merge.DecimIterable;
import ru.yandex.stockpile.server.shard.merge.EmptyIterator;
import ru.yandex.stockpile.server.shard.merge.FilterAfterIterable;
import ru.yandex.stockpile.server.shard.merge.FilterBeforeIterable;
import ru.yandex.stockpile.server.shard.merge.Iterable;
import ru.yandex.stockpile.server.shard.merge.Iterator;
import ru.yandex.stockpile.server.shard.merge.MergeIterable;
import ru.yandex.stockpile.server.shard.merge.OneItemIterator;

/**
 * @author Vladimir Gordiychuk
 */
public class CacheMetricArchiveMutable extends MetricArchiveMutable {
    private static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(CacheMetricArchiveMutable.class);
    private static final long FRAME_INTERVAL_MILLIS = TimeUnit.HOURS.toMillis(2L);
    private long latestEndFrameTs = 0;
    private final long decimatedAt;

    public CacheMetricArchiveMutable() {
        this.decimatedAt = 0;
    }

    public CacheMetricArchiveMutable(MetricHeader header, long decimatedAt) {
        super(header);
        this.decimatedAt = decimatedAt;
    }

    @Override
    protected long getSelfSize() {
        return SELF_SIZE;
    }

    /**
     * @param fromTsMillis inclusive
     * @param toMillis exclusive
     */
    public MetricSnapshot snapshot(long fromTsMillis, long toMillis) {
        var compressed = getCompressedDataRaw();
        if (toMillis == 0) {
            toMillis = Long.MAX_VALUE;
        }

        // fast path
        if (fromTsMillis > latestEndFrameTs) {
            // Buffer should be copied, because source buffer can be changed caused by close frame
            var lastCompressed = compressed.copy(getLatestFramePos(), compressed.readableBits() - getLatestFramePos());
            var latestFrame = latestFrame(lastCompressed, fromTsMillis, toMillis);
            return new MetricSnapshot(header(), decim(latestFrame), lastCompressed::release);
        }

        compressed.retain();
        Iterable closed = findClosed(compressed, fromTsMillis, toMillis);
        if (toMillis < latestEndFrameTs && isSorted()) {
            return new MetricSnapshot(header(), decim(closed), compressed::release);
        }

        // Buffer should be copied, because source buffer can be changed caused by close frame
        var lastCompressed = compressed.copy(getLatestFramePos(), compressed.readableBits() - getLatestFramePos());
        Iterable last = latestFrame(lastCompressed, fromTsMillis, toMillis);
        var result = MergeIterable.of(List.of(closed, last));

        return new MetricSnapshot(header(), decim(result), () -> {
            compressed.release();
            lastCompressed.release();
        });
    }

    @Override
    public boolean closeFrame() {
        if (frameRecordCount() < 1000) {
            return false;
        }

        long past = getLastTsMillis() - latestEndFrameTs;
        if (past < FRAME_INTERVAL_MILLIS) {
            return false;
        }

        if (super.closeFrame()) {
            latestEndFrameTs = getLastTsMillis();
            return true;
        }

        return false;
    }

    @Override
    public void forceCloseFrame() {
        super.forceCloseFrame();
        latestEndFrameTs = getLastTsMillis();
    }

    private Iterable latestFrame(BitBuf copy, long fromMillis, long toMillis) {
        boolean sorted = isSorted();
        boolean duplicates = hasDuplicates();
        if (!sorted || duplicates) {
            return new LastUnsorted(getFormat(), getType(), columnSetMask(), copy, fromMillis, toMillis, frameRecordCount(), sorted, duplicates);
        }

        Iterable it = new LastSorted(getFormat(), getType(), columnSetMask(), copy, frameRecordCount(), latestEndFrameTs + 1, getLastTsMillis());
        if (fromMillis > latestEndFrameTs) {
            it = new FilterBeforeIterable(it, fromMillis);
        }
        if (toMillis <= getLastTsMillis()) {
            it = new FilterAfterIterable(it, toMillis);
        }
        return it;
    }

    private Iterable decim(Iterable source) {
        DecimPolicy policy = DecimPoliciesPredefined.policyByNumber(decimPolicyId);
        return new DecimIterable(source, policy, System.currentTimeMillis(), decimatedAt);
    }

    private Iterable findClosed(BitBuf compressed, long fromTsMillis, long toTsMillis) {
        // Don't copy source buffer because all modify operations allocate new one,
        // so we can avoid allocate new buffer to read points from it
        var sliced = compressed.slice(0, getLatestFramePos());
        int records = getRecordCount() - frameRecordCount();
        Iterable result = new ClosedArchiveIterable(getFormat(), getType(), columnSetMask(), records, sliced);
        if (fromTsMillis != 0) {
            result = new FilterBeforeIterable(result, fromTsMillis);
        }

        if (toTsMillis <= latestEndFrameTs) {
            result = new FilterAfterIterable(result, toTsMillis);
        }
        return result;
    }

    private static class LastUnsorted implements Iterable {
        private final StockpileFormat format;
        private final MetricType type;
        private final int mask;
        private final int records;
        private final BitBuf compressed;
        private final long fromMillis;
        private final long toMillis;
        private final boolean sorted;
        private final boolean hasDuplicates;

        public LastUnsorted(StockpileFormat format, MetricType type, int mask, BitBuf compressed, long fromTsMillis, long toMillis, int records, boolean sorted, boolean hasDuplicates) {
            this.format = format;
            this.type = type;
            this.mask = mask;
            this.compressed = compressed;
            this.fromMillis = fromTsMillis;
            this.toMillis = toMillis;
            this.records = records;
            this.sorted = sorted;
            this.hasDuplicates = hasDuplicates;
        }

        @Override
        public Iterator iterator() {
            var list = unpack();
            var cropped = list.view().cropForResponse(fromMillis, toMillis - 1, false);
            if (cropped.isEmpty()) {
                return EmptyIterator.INSTANCE;
            }

            long firstTsMillis = cropped.getTsMillis(0);
            long lastTsMillis = cropped.getTsMillis(cropped.length() - 1);
            return OneItemIterator.of(cropped.iterator(), type, firstTsMillis, lastTsMillis, compressed.bytesSize());
        }

        private AggrGraphDataArrayList unpack() {
            var list = AggrGraphDataArrayList.of(new BinaryAggrGraphDataListIterator(type, mask, compressed.asReadOnly(), records));
            if (sorted) {
                if (hasDuplicates) {
                    list.mergeAdjacent();
                }
                return list;
            }

            list.sortAndMerge();
            return list;
        }

        @Override
        public int columnSetMask() {
            return mask;
        }

        @Override
        public int elapsedRecords() {
            return records;
        }

        @Override
        public int elapsedBytes() {
            return compressed.bytesSize();
        }
    }

    private static class LastSorted implements Iterable {
        private final StockpileFormat format;
        private final MetricType type;
        private final int mask;
        private final int records;
        private final BitBuf compressed;
        private final long firstTsMilli;
        private final long lastTsMillis;

        public LastSorted(StockpileFormat format, MetricType type, int mask, BitBuf compressed, int records, long firstTsMilli, long lastTsMillis) {
            this.format = format;
            this.type = type;
            this.mask = mask;
            this.records = records;
            this.compressed = compressed;
            this.firstTsMilli = firstTsMilli;
            this.lastTsMillis = lastTsMillis;
        }

        @Override
        public Iterator iterator() {
            var it = new BinaryAggrGraphDataListIterator(type, mask, compressed.asReadOnly(), records);
            return OneItemIterator.of(it, type, firstTsMilli, lastTsMillis, records);
        }

        @Override
        public int columnSetMask() {
            return mask;
        }

        @Override
        public int elapsedRecords() {
            return records;
        }

        @Override
        public int elapsedBytes() {
            return compressed.bytesSize();
        }
    }
}
