package ru.yandex.stockpile.server.shard.cache;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.concurrent.NotThreadSafe;

import ru.yandex.solomon.memory.layout.MemMeasurable;
import ru.yandex.solomon.memory.layout.MemoryCounter;
import ru.yandex.stockpile.client.shard.StockpileMetricId;
import ru.yandex.stockpile.server.cache.Long2ObjectLruCache;
import ru.yandex.stockpile.server.data.log.StockpileLogEntryContent;

/**
 * @author Stepan Koltsov
 */
@NotThreadSafe
@ParametersAreNonnullByDefault
public class MetricDataCache implements MemMeasurable, AutoCloseable {
    private static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(MetricDataCache.class);

    private final int shardId;
    private final Long2ObjectLruCache<MetricDataCacheEntry> cache = new Long2ObjectLruCache<>();

    private long hit;
    private long miss;

    public MetricDataCache(int shardId) {
        this.shardId = shardId;
    }

    @Override
    public long memorySizeIncludingSelf() {
        return SELF_SIZE + cache.memorySizeIncludingSelf();
    }

    public void setMaxSizeBytes(long cacheSize) {
        cache.setMaxSizeBytes(cacheSize);
    }

    public long getMaxSizeBytes() {
        return cache.getMaxSizeBytes();
    }

    public long getBytesUsed() {
        return cache.getBytesUsed();
    }

    public long getRecordsUsed() {
        return cache.getRecordsUsed();
    }

    public long getAddedRecords() {
        return cache.getAddedRecords();
    }

    public long getRemovedRecords() {
        return cache.getRemovedRecords();
    }

    public long getAddedBytes() {
        return cache.getAddedBytes();
    }

    public long getRemovedBytes() {
        return cache.getRemovedBytes();
    }

    public long getHit() {
        return hit;
    }

    public long getMiss() {
        return miss;
    }

    public int size() {
        return cache.size();
    }

    public boolean isEnabled() {
        return cache.getMaxSizeBytes() != 0;
    }

    @Nullable
    public MetricSnapshot getSnapshot(long localId) {
        return getSnapshot(localId, 0);
    }

    @Nullable
    public MetricSnapshot getSnapshot(long localId, long fromMillis) {
        return getSnapshot(localId, fromMillis, Long.MAX_VALUE);
    }

    @Nullable
    public MetricSnapshot getSnapshot(long localId, long fromMillis, long toMillis) {
        MetricDataCacheEntry entry = cache.get(localId);
        if (entry != null) {
            if (!entry.isInitializedWithData()) {
                miss++;
                return null;
            }

            if (!entry.isLoaded(fromMillis)) {
                miss++;
                cache.replace(localId, new MetricDataCacheEntry());
                return null;
            }

            hit++;
            return entry.snapshot(fromMillis, toMillis);
        }

        miss++;
        cache.replace(localId, new MetricDataCacheEntry());
        return null;
    }

    public void updateWithOnWriteCompleted(StockpileLogEntryContent logEntryContent) {
        if (!isEnabled()) {
            return;
        }

        long bytesDiff = logEntryContent.getDataByMetricId()
                .long2ObjectEntrySet()
                .parallelStream()
                .mapToLong(entry -> {
                    long localId = entry.getLongKey();
                    MetricDataCacheEntry cacheEntry = cache.get(localId);
                    if (cacheEntry == null) {
                        return 0;
                    }

                    final long prevSizeBytes = cacheEntry.memorySizeIncludingSelf();
                    try {
                        cacheEntry.updateWithOrWriteCompleted(shardId, entry.getLongKey(), entry.getValue());
                    } catch (Exception e) {
                        throw new RuntimeException("cannot update metric " + StockpileMetricId.toString(shardId, entry.getLongKey()), e);
                    }
                    return cacheEntry.memorySizeIncludingSelf() - prevSizeBytes;
                })
                .sum();
        cache.sizeUpdated(bytesDiff);
        cache.removeEntriesAboveSizeLimit();
    }

    public void readCompleted(long localId, MetricDataCacheEntry loadedEntry) {
        MetricDataCacheEntry inCache = cache.get(localId);
        if (inCache == null) {
            // already evicted
            return;
        }

        final int prevSizeBytes = inCache.memorySizeIncludingSelfInt();
        inCache.updateAfterReadCompleted(loadedEntry);
        cache.sizeUpdated(inCache.memorySizeIncludingSelfInt() - prevSizeBytes);
        cache.removeEntriesAboveSizeLimit();
    }

    public void flush() {
        cache.clear();
    }

    public void removeEntry(long localId) {
        cache.remove(localId);
    }

    @Override
    public void close() {
        cache.close();
    }
}
