package ru.yandex.stockpile.server.shard.cache;

import java.util.concurrent.TimeUnit;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.WillClose;
import javax.annotation.WillNotClose;

import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.codec.archive.header.MetricHeader;
import ru.yandex.solomon.memory.layout.MemMeasurable;
import ru.yandex.solomon.memory.layout.MemoryCounter;
import ru.yandex.solomon.model.point.RecyclableAggrPoint;
import ru.yandex.solomon.model.timeseries.AggrGraphDataListIterator;
import ru.yandex.solomon.model.timeseries.decim.DecimPoliciesPredefined;
import ru.yandex.stockpile.server.shard.merge.AggrPointIteratorWrapper;
import ru.yandex.stockpile.server.shard.merge.DecimIterator;
import ru.yandex.stockpile.server.shard.merge.Iterator;

import static ru.yandex.stockpile.server.shard.MetricArchives.typeSafeAppend;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class MetricDataCacheEntry implements MemMeasurable, AutoCloseable {
    private static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(MetricDataCacheEntry.class);
    private static final long INIT_FRAME_INTERVAL_MILLIS = TimeUnit.DAYS.toMillis(1L);

    private CacheMetricArchiveMutable archive;
    private boolean wasInitializedWithData = false;
    private long fromMillis;

    public MetricDataCacheEntry() {
        this.archive = new CacheMetricArchiveMutable();
        this.fromMillis = System.currentTimeMillis();
    }

    public MetricDataCacheEntry(long fromMillis, MetricHeader header, long decimatedAt, AggrGraphDataListIterator it) {
        long expectedNextFrameTs = fromMillis + INIT_FRAME_INTERVAL_MILLIS;
        this.archive = new CacheMetricArchiveMutable(header, decimatedAt);
        this.wasInitializedWithData = true;
        this.fromMillis = fromMillis;

        final int mask = it.columnSetMask();
        archive.ensureBytesCapacity(mask, it.estimatePointsCount() * 2);
        var point = RecyclableAggrPoint.newInstance();
        try {
            while (it.next(point)) {
                if (point.tsMillis >= expectedNextFrameTs) {
                    archive.closeFrame();
                    expectedNextFrameTs = archive.getLastTsMillis() + INIT_FRAME_INTERVAL_MILLIS;
                }
                archive.addRecord(point);
            }
        } finally {
            point.recycle();
        }
    }

    public static MetricDataCacheEntry of(long fromMillis, MetricHeader header, Iterator source) {
        var now = System.currentTimeMillis();
        var policy = DecimPoliciesPredefined.policyByNumber(header.getDecimPolicyId());
        var decimated = DecimIterator.of(source, policy, now, 0);
        return new MetricDataCacheEntry(fromMillis, header, now, new AggrPointIteratorWrapper(decimated));
    }

    public void updateWithOrWriteCompleted(int shardId, long localId, MetricArchiveMutable update) {
        typeSafeAppend(shardId, localId, archive, update);
        archive.closeFrame();
        wasInitializedWithData = true;
    }

    @Override
    public long memorySizeIncludingSelf() {
        return SELF_SIZE + archive.memorySizeIncludingSelf();
    }

    public void updateAfterReadCompleted(MetricDataCacheEntry entry) {
        this.archive = updateArchive(this.archive, entry.archive);
        entry.archive = null;
        this.wasInitializedWithData = true;
        this.fromMillis = entry.fromMillis;
    }

    private CacheMetricArchiveMutable updateArchive(@WillClose CacheMetricArchiveMutable local, @WillNotClose
        CacheMetricArchiveMutable update) {
        if (wasInitializedWithData) {
            // was wrote more fresh data during our load round trip,
            // it's save append to end of timeseries without merge it via iterators
            update.updateWith(local);
        }

        local.close();
        return update;
    }

    /**
     * @param fromMillis inclusive
     * @param toMillis exclusive
     */
    public MetricSnapshot snapshot(long fromMillis, long toMillis) {
        return archive.snapshot(fromMillis, toMillis);
    }

    public boolean isLoaded(long fromMillis) {
        return wasInitializedWithData && this.fromMillis <= fromMillis;
    }

    public boolean isInitializedWithData() {
        return wasInitializedWithData;
    }

    @Override
    public void close() {
        if (archive != null) {
            archive.close();
            archive = null;
        }
    }
}
