package ru.yandex.stockpile.server.shard.cache;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.OptionalLong;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.function.ToLongFunction;
import java.util.stream.Collectors;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.misc.actor.ActorRunner;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.solomon.staffOnly.annotations.LinkedOnRootPage;
import ru.yandex.solomon.staffOnly.html.AHref;
import ru.yandex.solomon.staffOnly.manager.ExtraContentParam;
import ru.yandex.solomon.staffOnly.manager.ManagerController;
import ru.yandex.solomon.staffOnly.manager.special.ExtraContent;
import ru.yandex.solomon.staffOnly.manager.special.InstantMillis;
import ru.yandex.solomon.staffOnly.manager.table.TableColumnDef;
import ru.yandex.solomon.staffOnly.manager.table.TableColumnDefImpl;
import ru.yandex.stockpile.server.shard.StockpileExecutor;
import ru.yandex.stockpile.server.shard.StockpileLocalShards;
import ru.yandex.stockpile.server.shard.StockpileMemory;
import ru.yandex.stockpile.server.shard.StockpileScheduledExecutor;
import ru.yandex.stockpile.server.shard.StockpileShard;

/**
 * @author Stepan Koltsov
 */
@Component
@LinkedOnRootPage("Cache Manager")
public class StockpileCacheManager implements AutoCloseable {
    private final StockpileLocalShards shards;
    private final ActorRunner actor;
    private final ScheduledFuture<?> future;
    @InstantMillis
    private long lastIterationInstant;

    @Autowired
    public StockpileCacheManager(
        StockpileLocalShards shards,
        @StockpileExecutor ExecutorService executor,
        @StockpileScheduledExecutor ScheduledExecutorService timer)
    {
        this.shards = shards;
        this.actor = new ActorRunner(this::updateCachePrefs, executor);
        this.future = timer.scheduleAtFixedRate(actor::schedule, 0, 5, TimeUnit.SECONDS);
    }

    @Override
    public void close() {
        future.cancel(false);
    }

    private enum WeightAddendum {
        SIZE(
            2,
            s -> s.recordCount().orElse(10 << 10),
            s -> {
                OptionalLong recordCount = s.recordCount();
                if (recordCount.isPresent()) {
                    return Long.toString(recordCount.getAsLong());
                } else {
                    return "?";
                }
            }),
        REQUESTS(
            5,
            s -> Math.round(s.metrics.read.avgMetricsRps.getRate(TimeUnit.SECONDS)),
            s -> String.format("%.2f", s.metrics.read.avgMetricsRps.getRate(TimeUnit.SECONDS))),
        MISSES(
            3,
            s -> Math.round(s.metrics.cache.avgMiss.getRate(TimeUnit.SECONDS)),
            s -> String.format("%.2f", s.metrics.cache.avgMiss.getRate(TimeUnit.SECONDS))
        )
        ;

        private final int weight;
        private final ToLongFunction<StockpileShard> get;
        private final Function<StockpileShard, String> string;

        WeightAddendum(int weight,
            ToLongFunction<StockpileShard> get,
            Function<StockpileShard, String> string)
        {
            this.weight = weight;
            this.get = get;
            this.string = string;
        }
    }

    public void updateCachePrefs() {
        StockpileShard[] localShards = shards.stream().toArray(StockpileShard[]::new);
        if (localShards.length == 0) {
            return;
        }

        long now = System.currentTimeMillis();
        updateShardCacheSizes(localShards);

        lastIterationInstant = now;
    }

    private void updateShardCacheSizes(StockpileShard[] localShards) {
        long[] shardCacheSizes = new long[localShards.length];

        CacheWeights.updateWeights(
            Arrays.asList(localShards),
            WeightAddendum.class,
            w -> w.weight,
            (shard, weightAddendum) -> weightAddendum.get.applyAsLong(shard),
            StockpileMemory.getTotalCacheMemoryLimit(),
            shardCacheSizes);

        for (int i = 0; i < localShards.length; ++i) {
            // TODO: clear cache if size reduced
            localShards[i].setMaxCacheSize(Math.max(shardCacheSizes[i], 1 << 20));
        }
    }

    @ExtraContent("Shard caches")
    private void extra(ExtraContentParam p) {
        ArrayList<TableColumnDef<StockpileShard>> columns = new ArrayList<>();
        columns.add(new TableColumnDefImpl<>("Id", s -> new AHref(ManagerController.namedObjectLink(s), s.shardId)));

        for (WeightAddendum weightAddendum : WeightAddendum.values()) {
            columns.add(new TableColumnDefImpl<>(
                weightAddendum.name() + "(" + weightAddendum.weight + ")",
                weightAddendum.string::apply));
        }

        columns.add(new TableColumnDefImpl<>("C max bytes", s -> DataSize.shortString(s.cache.getMaxSizeBytes())));
        columns.add(new TableColumnDefImpl<>("C used bytes", s -> DataSize.shortString(s.cache.getBytesUsed())));
        columns.add(new TableColumnDefImpl<>("C entries", s -> s.cache.getRecordsUsed()));

        List<StockpileShard> shards = this.shards.stream().collect(Collectors.toList());
        p.managerWriter().listTable(shards, columns);
    }
}
