package ru.yandex.stockpile.server.shard.iter;

import java.util.Deque;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentLinkedDeque;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicIntegerFieldUpdater;
import java.util.concurrent.atomic.AtomicLong;

import javax.annotation.CheckReturnValue;
import javax.annotation.Nullable;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.memory.layout.MemoryCounter;
import ru.yandex.stockpile.server.data.chunk.ChunkWithNo;
import ru.yandex.stockpile.server.data.chunk.SnapshotAddress;
import ru.yandex.stockpile.server.shard.load.AsyncIterator;

/**
 * @author Vladimir Gordiychuk
 */
public abstract class KvSnapshotChunkIterator implements AsyncIterator<ChunkWithNo> {
    public static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(KvSnapshotChunkIterator.class);
    private static final AtomicIntegerFieldUpdater<KvSnapshotChunkIterator> nextChunkNoUpdater =
        AtomicIntegerFieldUpdater.newUpdater(KvSnapshotChunkIterator.class, "nextChunkNo");

    private final SnapshotAddress snapshotAddress;
    private final int chunkCount;
    private final AtomicLong memoryUse = new AtomicLong(MemoryCounter.CompletableFuture_SELF_SIZE);
    private final AtomicInteger prefetchSize = new AtomicInteger(0);
    private final Deque<ChunkWithNo> prefetched = new ConcurrentLinkedDeque<>();
    private volatile int nextChunkNo = 0;
    private volatile CompletableFuture<Void> next = new CompletableFuture<>();

    public KvSnapshotChunkIterator(SnapshotAddress snapshotAddress, int chunkCount) {
        this.snapshotAddress = snapshotAddress;
        this.chunkCount = chunkCount;
    }

    protected abstract CompletableFuture<ChunkWithNo> readNext(SnapshotAddress snapshotAddress, int chunkNo);

    private void readNextChunk(int chunkNo) {
        if (chunkNo >= chunkCount) {
            next.complete(null);
            return;
        }

        var copy = this.next;
        var task = readNext(snapshotAddress, chunkNo)
            .thenAccept(chunk -> {
                memoryUse.addAndGet(chunk.memorySizeIncludingSelf());
                prefetched.addLast(chunk);
                scheduleNextFetch(chunk.getNo());
            });
        CompletableFutures.whenComplete(task, copy);
    }

    @Override
    @CheckReturnValue
    public CompletableFuture<ChunkWithNo> next() {
        CompletableFuture<ChunkWithNo> result = new CompletableFuture<>();
        next(result);
        return result;
    }

    private void next(CompletableFuture<ChunkWithNo> doneFuture) {
        var future = this.next;
        ChunkWithNo local = pollFetched();
        if (local != null) {
            doneFuture.complete(local);
            return;
        }

        if (nextChunkNo == 0) {
            readNextChunk(0);
        }

        future.whenComplete((r, e) -> {
            if (e != null) {
                doneFuture.completeExceptionally(e);
                return;
            }

            if (nextChunkNo >= chunkCount) {
                doneFuture.complete(pollFetched());
                return;
            }

            // consumer can process chunk faster, prefetch more chunks at once
            if (prefetched.size() >= prefetchSize.get()) {
                prefetchSize.incrementAndGet();
            }

            next(doneFuture);
        });
    }

    @Nullable
    private ChunkWithNo pollFetched() {
        ChunkWithNo next = prefetched.poll();
        if (next != null) {
            memoryUse.addAndGet(-next.memorySizeIncludingSelf());
            ChunkWithNo last = prefetched.peekLast();
            if (last == null) {
                last = next;
            }
            scheduleNextFetch(last.getNo());
        }
        return next;
    }

    private void scheduleNextFetch(int chunkNo) {
        // schedule next chunk fetching only once
        if (prefetchSize.get() <= prefetched.size()) {
            return;
        }

        int nextChunkNo = chunkNo + 1;
        if (nextChunkNoUpdater.compareAndSet(this, chunkNo, nextChunkNo)) {
            next = new CompletableFuture<>();
            readNextChunk(nextChunkNo);
        }
    }

    @Override
    public long memorySizeIncludingSelf() {
        return memoryUse.get();
    }
}
