package ru.yandex.stockpile.server.shard.iter;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.CompletableFuture;

import javax.annotation.Nullable;

import ru.yandex.solomon.memory.layout.MemoryCounter;
import ru.yandex.stockpile.client.shard.StockpileLocalId;
import ru.yandex.stockpile.kikimrKv.counting.ReadClass;
import ru.yandex.stockpile.memState.MetricIdAndData;
import ru.yandex.stockpile.server.data.dao.StockpileShardStorage;
import ru.yandex.stockpile.server.data.index.SnapshotIndex;
import ru.yandex.stockpile.server.shard.ShardThread;
import ru.yandex.stockpile.server.shard.load.AsyncIterator;

/**
 * @author Vladimir Gordiychuk
 */
public class MetricsIterator implements AsyncIterator<List<MetricIdAndData>> {
    private static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(MetricsIterator.class);
    private static final long LOCAL_ID_EOF = -1;
    private static final long LOCAL_ID_UNINIT = 0;

    private AsyncIterator<MetricIdAndData>[] snapshotIterators;
    private long[] currentLocalIdPerIterator;
    private MetricIdAndData[] metrics;

    public MetricsIterator(StockpileShardStorage storage, ShardThread shardThread, SnapshotIndex[] indexes, ReadClass readClass) {
        this(Arrays.stream(indexes)
            .filter(i -> !i.isEmpty())
            .map(index -> new SnapshotIterator(storage, shardThread, index, readClass))
            .toArray(SnapshotIterator[]::new));
    }

    public MetricsIterator(AsyncIterator<MetricIdAndData>[] iterators) {
        this.snapshotIterators = iterators;
        this.currentLocalIdPerIterator = new long[iterators.length];
        this.metrics = new MetricIdAndData[iterators.length];
    }

    public CompletableFuture<List<MetricIdAndData>> next() {
        CompletableFuture<List<MetricIdAndData>> doneFuture = new CompletableFuture<>();
        fillGapInIterator(0, doneFuture);
        return doneFuture;
    }

    private void fillGapInIterator(int iteratorIdx, CompletableFuture<List<MetricIdAndData>> doneFuture) {
        try {
            for (; iteratorIdx < snapshotIterators.length; iteratorIdx++) {
                final long localId = currentLocalIdPerIterator[iteratorIdx];
                if (localId == LOCAL_ID_EOF) {
                    // skip already drained iterator
                    continue;
                }

                if (localId != LOCAL_ID_UNINIT) {
                    // skip already fetched iterator
                    continue;
                }

                // fetch next snapshot from this iterator
                var snapshotIterator = snapshotIterators[iteratorIdx];

                CompletableFuture<MetricIdAndData> nextFuture = snapshotIterator.next();
                if (nextFuture.isDone()) {
                    MetricIdAndData next = nextFuture.getNow(null);
                    processNext(iteratorIdx, next);
                } else {
                    final int finalIteratorIdx = iteratorIdx;
                    nextFuture.whenComplete((next, throwable) -> {
                        if (throwable != null) {
                            doneFuture.completeExceptionally(throwable);
                            return;
                        }

                        try {
                            processNext(finalIteratorIdx, next);
                        } catch (Throwable e) {
                            doneFuture.completeExceptionally(e);
                            return;
                        }

                        // continue cycle from the next position
                        fillGapInIterator(finalIteratorIdx + 1, doneFuture);
                    });

                    // cycle will be continued when async operation complete
                    return;
                }
            }

            doneFuture.complete(collectMetrics());
        } catch (Throwable e) {
            doneFuture.completeExceptionally(e);
        }
    }

    @Nullable
    private List<MetricIdAndData> collectMetrics() {
        long minLocalId = StockpileLocalId.min(currentLocalIdPerIterator);
        if (minLocalId == LOCAL_ID_EOF) {
            return null;
        }

        if (minLocalId == LOCAL_ID_UNINIT) {
            throw new IllegalStateException();
        }

        List<MetricIdAndData> result = new ArrayList<>(currentLocalIdPerIterator.length);
        for (int index = 0; index < currentLocalIdPerIterator.length; index++) {
            if (minLocalId != currentLocalIdPerIterator[index]) {
                continue;
            }

            MetricIdAndData data = metrics[index];
            currentLocalIdPerIterator[index] = LOCAL_ID_UNINIT;
            metrics[index] = null;
            result.add(data);
        }

        return result;
    }

    private void processNext(int iteratorIdx, @Nullable MetricIdAndData next) {
        if (next == null) {
            currentLocalIdPerIterator[iteratorIdx] = LOCAL_ID_EOF;
            metrics[iteratorIdx] = null;
        } else {
            currentLocalIdPerIterator[iteratorIdx] = next.localId();
            metrics[iteratorIdx] = next;
        }
    }

    @Override
    public long memorySizeIncludingSelf() {
        return SELF_SIZE
            + MemoryCounter.arrayObjectSizeWithContent(metrics)
            + MemoryCounter.arrayObjectSize(currentLocalIdPerIterator)
            + MemoryCounter.arrayObjectSizeWithContent(snapshotIterators);
    }
}
