package ru.yandex.stockpile.server.shard.iter;

import java.util.concurrent.CompletableFuture;

import javax.annotation.CheckReturnValue;
import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.memory.layout.MemoryCounter;
import ru.yandex.stockpile.kikimrKv.counting.ReadClass;
import ru.yandex.stockpile.memState.MetricIdAndData;
import ru.yandex.stockpile.server.data.chunk.ChunkAddressGlobal;
import ru.yandex.stockpile.server.data.chunk.ChunkIterator;
import ru.yandex.stockpile.server.data.chunk.ChunkWithNo;
import ru.yandex.stockpile.server.data.dao.StockpileShardStorage;
import ru.yandex.stockpile.server.data.index.SnapshotIndex;
import ru.yandex.stockpile.server.shard.ShardThread;
import ru.yandex.stockpile.server.shard.load.AsyncIterator;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class SnapshotIterator implements AsyncIterator<MetricIdAndData> {
    public static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(SnapshotIterator.class);

    private final SnapshotIndex index;
    private final AsyncIterator<ChunkWithNo> chunkIterator;
    private boolean completed;
    private ChunkIterator currentChunkIterator;

    public SnapshotIterator(StockpileShardStorage storage, ShardThread mergeProcess, SnapshotIndex index, ReadClass readClass) {
        this(index, new SnapshotChunkIteratorPrefetch(storage, readClass, mergeProcess, index.snapshotAddress(), index.getContent().getChunksCount()));
    }

    public SnapshotIterator(SnapshotIndex index, AsyncIterator<ChunkWithNo> chunkIterator) {
        this.index = index;
        this.chunkIterator = chunkIterator;
    }

    private boolean hasNextLocal() {
        return currentChunkIterator != null && currentChunkIterator.hasNext();
    }

    public long size() {
        return index.getContent().getMetricCount();
    }

    @Nonnull
    private MetricIdAndData nextLocal() {
        return currentChunkIterator.next();
    }

    @Override
    public long memorySizeIncludingSelf() {
        return SELF_SIZE + chunkIterator.memorySizeIncludingSelf();
    }

    @CheckReturnValue
    public CompletableFuture<MetricIdAndData> next() {
        try {
            if (hasNextLocal()) {
                return completedFuture(nextLocal());
            }

            if (completed) {
                return completedFuture(null);
            }

            return chunkIterator.next()
                .thenApply(nextChunk -> {
                    if (nextChunk == null) {
                        currentChunkIterator = null;
                        completed = true;
                        return null;
                    } else {
                        currentChunkIterator = makeIterator(nextChunk);
                        if (currentChunkIterator.hasNext()) {
                            return currentChunkIterator.next();
                        } else {
                            throw new IllegalStateException(index.snapshotAddress() + " " +  nextChunk + " no metrics");
                        }
                    }
                });
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    private ChunkIterator makeIterator(ChunkWithNo nextChunk) {
        var address = new ChunkAddressGlobal(index.snapshotAddress(), nextChunk.getNo());
        try {
            return new ChunkIterator(
                address,
                index.getContent().getChunk(nextChunk.getNo()),
                index.getContent().getDecimatedAt(),
                nextChunk.getContent(),
                index.getContent().getFormat());
        } catch (Throwable e) {
            throw new RuntimeException(address.toString(), e);
        }
    }

    @Override
    public String toString() {
        return "SnapshotIterator{" +
            "index=" + index +
            ", chunkIterator=" + chunkIterator +
            ", currentChunkIterator=" + currentChunkIterator +
            '}';
    }
}
