package ru.yandex.stockpile.server.shard.load;

import java.util.concurrent.CompletableFuture;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;

import javax.annotation.Nullable;

import ru.yandex.misc.concurrent.CompletableFutures;


/**
 * @author Sergey Polovko
 */
public final class Async {
    private Async() {}


    public static <T> CompletableFuture<Void> forEach(AsyncIterator<T> it, Consumer<T> consumer) {
        return new SyncConsumeLoop<>(it, consumer).getDoneFuture();
    }

    public static <T> CompletableFuture<Void> forEachAsync(
        AsyncIterator<T> it,
        Function<T, CompletableFuture<Void>> consumer)
    {
        return new AsyncConsumeLoop<>(it, consumer).getDoneFuture();
    }

    /**
     * Iterate over async iterator and execute sync consumer for each element.
     */
    private static final class SyncConsumeLoop<T> implements BiConsumer<T, Throwable> {
        private final CompletableFuture<Void> doneFuture = new CompletableFuture<>();
        private final AsyncIterator<T> it;
        private final Consumer<T> consumer;

        SyncConsumeLoop(AsyncIterator<T> it, Consumer<T> consumer) {
            this.it = it;
            this.consumer = consumer;
            iteration();
        }

        CompletableFuture<Void> getDoneFuture() {
            return doneFuture;
        }

        /**
         * Synchronously iterate over ready elements, or subscribe on not ready one.
         */
        private void iteration() {
            while (!doneFuture.isDone()) {
                CompletableFuture<T> nextFuture = it.next();
                if (CompletableFutures.isCompletedSuccessfully(nextFuture)) {
                    safeConsume(nextFuture.getNow(null));
                } else {
                    nextFuture.whenComplete(this);
                    break;
                }
            }
        }

        /**
         * Continue loop after async data received.
         */
        @Override
        public void accept(T next, Throwable throwable) {
            if (throwable != null) {
                doneFuture.completeExceptionally(throwable);
            } else {
                safeConsume(next);
                iteration();
            }
        }

        /**
         * Run consumer and catch all possible exceptions.
         */
        private void safeConsume(@Nullable T next) {
            if (next == null) {
                doneFuture.complete(null);
                return;
            }

            try {
                consumer.accept(next);
            } catch (Throwable t) {
                doneFuture.completeExceptionally(t);
            }
        }
    }

    /**
     * Iterate over async iterator and execute async consumer for each element.
     */
    private static final class AsyncConsumeLoop<T> implements BiConsumer<T, Throwable> {
        private final CompletableFuture<Void> doneFuture = new CompletableFuture<>();
        private final AsyncIterator<T> it;
        private final Function<T, CompletableFuture<Void>> consumer;

        AsyncConsumeLoop(AsyncIterator<T> it, Function<T, CompletableFuture<Void>> consumer) {
            this.it = it;
            this.consumer = consumer;
            iteration();
        }

        CompletableFuture<Void> getDoneFuture() {
            return doneFuture;
        }

        /**
         * Synchronously iterate over ready elements, or subscribe on not ready one.
         */
        private void iteration() {
            while (!doneFuture.isDone()) {
                CompletableFuture<T> nextFuture = it.next();
                if (CompletableFutures.isCompletedSuccessfully(nextFuture)) {
                    T next = nextFuture.getNow(null);
                    if (!safeConsume(next)) {
                        break;
                    }
                } else {
                    nextFuture.whenComplete(this);
                    break;
                }
            }
        }

        /**
         * Continue loop after async data received.
         */
        @Override
        public void accept(T next, Throwable throwable) {
            if (throwable != null) {
                doneFuture.completeExceptionally(throwable);
                return;
            }

            if (safeConsume(next)) {
                iteration();
            }
        }

        /**
         * Run consumer and catch all possible exceptions.
         *
         * @return {@code true} if next iteration can be executed synchronously,
         *         {@code false} otherwise.
         */
        private boolean safeConsume(@Nullable T next) {
            if (next == null) {
                doneFuture.complete(null);
                return false;
            }

            try {
                CompletableFuture<Void> f = consumer.apply(next);
                if (CompletableFutures.isCompletedSuccessfully(f)) {
                    return true;
                }

                f.whenComplete((aVoid, throwable) -> {
                    if (throwable != null) {
                        doneFuture.completeExceptionally(throwable);
                    } else {
                        iteration();
                    }
                });
                return false;
            } catch (Throwable t) {
                doneFuture.completeExceptionally(t);
                return false;
            }
        }
    }
}
