package ru.yandex.stockpile.server.shard.load;

import java.util.Arrays;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.concurrent.Immutable;

import it.unimi.dsi.fastutil.ints.Int2LongMaps;
import it.unimi.dsi.fastutil.longs.Long2ObjectMaps;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.kikimr.client.kv.KikimrKvClient.KvEntryWithStats;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.config.protobuf.stockpile.EInvalidArchiveStrategy;
import ru.yandex.stockpile.server.Txn;
import ru.yandex.stockpile.server.data.DeletedShardSet;
import ru.yandex.stockpile.server.data.log.StockpileLogEntryContent;
import ru.yandex.stockpile.server.data.log.StockpileLogEntryContentImmutable;
import ru.yandex.stockpile.server.data.log.StockpileLogEntryContentImmutableSerializer;
import ru.yandex.stockpile.server.data.names.FileNameParsed;
import ru.yandex.stockpile.server.data.names.HasTxn;


/**
 * <p>Groups multiple files ({@link KvEntryWithStats}) with logs by their transaction numbers (txn)
 * and performs parsing of according chunks into single {@link StockpileLogEntryContent}.
 *
 * <p>Parsing performed in thread from the given {@link Executor}.
 *
 * <p>This class uses COW to maintain current state (batch of files with same txn which must be
 * parsed together) to allow to use this iterator from different threads.
 *
 * @author Sergey Polovko
 */
public class KvLogParsingIterator implements AsyncIterator<StockpileLogEntryContentImmutable> {
    private static final Logger logger = LoggerFactory.getLogger(KvLogParsingIterator.class);

    private final Executor executor;
    private final EInvalidArchiveStrategy strategy;
    private final AsyncIterator<KvEntryWithStats> readIterator;
    private final AtomicReference<State> state = new AtomicReference<>(EmptyState.INSTANCE);
    private final AtomicLong totalSize = new AtomicLong(0);

    public KvLogParsingIterator(Executor executor, EInvalidArchiveStrategy strategy, AsyncIterator<KvEntryWithStats> readIterator) {
        this.executor = executor;
        this.strategy = strategy;
        this.readIterator = readIterator;
    }

    @Override
    public CompletableFuture<StockpileLogEntryContentImmutable> next() {
        while (true) {
            if (state.get() == null) {
                // done
                return CompletableFuture.completedFuture(null);
            }

            // get the next file from underlying iterator.
            // if the file was already read or iterator is ended then
            // process it synchronously
            CompletableFuture<KvEntryWithStats> nextFuture = readIterator.next();
            if (CompletableFutures.isCompletedSuccessfully(nextFuture)) {
                try {
                    KvEntryWithStats next = nextFuture.getNow(null);
                    CompletableFuture<StockpileLogEntryContentImmutable> logEntry = processFile(next);
                    if (logEntry != null) {
                        return logEntry;
                    }
                } catch (Throwable t) {
                    return CompletableFuture.failedFuture(t);
                }

                // file has same txn as previously received file, so continue to
                // collect all these files in one state
                continue;
            }

            // continue this loop when file will be read
            return nextFuture.thenComposeAsync(next -> {
                CompletableFuture<StockpileLogEntryContentImmutable> logEntry = processFile(next);
                if (logEntry != null) {
                    return logEntry;
                }
                return next();
            }, executor);
        }
    }

    public long getTotalSize() {
        return totalSize.get();
    }

    /**
     * @return future if parsing scheduled or {@code null} otherwise.
     */
    @Nullable
    private CompletableFuture<StockpileLogEntryContentImmutable> processFile(KvEntryWithStats file) {
        if (file == null) {
            // underlying iterator is ended, so flip state (current -> null) to mark
            // this iterator as ended and schedule parsing of the last files in the state
            State state = this.state.get();
            if (this.state.compareAndSet(state, null) && state.txn() != Txn.INVALID) {
                return parseFiles(state);
            }
        } else {
            // add this file in current state if its txn is the same as in previous file,
            // and schedule parsing
            long txn = parseTxn(file.getName());
            Txn.validateTxn(txn);

            State prevState, nextState;
            do {
                prevState = state.get();
                nextState = prevState.add(txn, file);
            } while (!state.compareAndSet(prevState, nextState));

            if (prevState.txn() != Txn.INVALID && prevState.txn() != nextState.txn()) {
                return parseFiles(prevState);
            }
        }
        return null;
    }

    private CompletableFuture<StockpileLogEntryContentImmutable> parseFiles(State state) {
        return CompletableFuture.supplyAsync(() -> {
                long filesSize = 0;
                for (byte[] chunk : state.content()) {
                    filesSize += chunk.length;
                }
                totalSize.addAndGet(filesSize);
                try {
                    return StockpileLogEntryContentImmutableSerializer.S.deserializeParts(state.content());
                } catch (Throwable e) {
                    if (strategy == EInvalidArchiveStrategy.DROP) {
                        logger.error("Failed to parse file " + state, e);
                        return new StockpileLogEntryContentImmutable(Long2ObjectMaps.emptyMap(), new DeletedShardSet(), Int2LongMaps.EMPTY_MAP);
                    } else {
                        throw new RuntimeException("Failed to parse file " + state, e);
                    }
                }
            },
            executor);
    }

    private static long parseTxn(String name) {
        FileNameParsed parsed = FileNameParsed.parseCurrent(name);
        if (parsed instanceof HasTxn) {
            return ((HasTxn) parsed).txn();
        }

        throw new IllegalStateException("cannot parse txn from file \'" + name + '\'');
    }

    /**
     * STATE
     */
    @Immutable
    private interface State {
        long txn();
        State add(long txn, KvEntryWithStats entry);
        byte[][] content();
    }

    /**
     * EMPTY STATE
     */
    private static final class EmptyState implements State {
        static final EmptyState INSTANCE = new EmptyState();

        @Override
        public long txn() {
            return Txn.INVALID;
        }

        @Override
        public State add(long txn, KvEntryWithStats entry) {
            return new SingleEntryState(txn, entry);
        }

        @Override
        public byte[][] content() {
            return new byte[0][];
        }

        @Override
        public String toString() {
            return "{}";
        }
    }

    /**
     * SINGLE ENTRY STATE
     */
    private static final class SingleEntryState implements State {
        private final long txn;
        private final KvEntryWithStats entry;

        SingleEntryState(long txn, KvEntryWithStats entry) {
            this.txn = txn;
            this.entry = entry;
        }

        @Override
        public long txn() {
            return txn;
        }

        @Override
        public State add(long txn, KvEntryWithStats entry) {
            if (this.txn == txn) {
                return new MultiEntriesState(txn, this.entry, entry);
            }
            return new SingleEntryState(txn, entry);
        }

        @Override
        public byte[][] content() {
            return new byte[][] { entry.getValue() };
        }

        @Override
        public String toString() {
            return "{txn=" + txn + ", entry=" + entry.getName() + '}';
        }
    }

    /**
     * MULTI ENTRIES STATE
     */
    private static final class MultiEntriesState implements State {
        private final long txn;
        private final KvEntryWithStats[] entries;

        MultiEntriesState(long txn, KvEntryWithStats... entries) {
            this.txn = txn;
            this.entries = entries;
        }

        @Override
        public long txn() {
            return txn;
        }

        @Override
        public State add(long txn, KvEntryWithStats entry) {
            if (this.txn == txn) {
                KvEntryWithStats[] entries = new KvEntryWithStats[this.entries.length + 1];
                System.arraycopy(this.entries, 0, entries, 0, this.entries.length);
                entries[entries.length - 1] = entry;
                return new MultiEntriesState(txn, entries);
            }
            return new SingleEntryState(txn, entry);
        }

        @Override
        public byte[][] content() {
            byte[][] content = new byte[entries.length][];
            for (int i = 0; i < entries.length; i++) {
                content[i] = entries[i].getValue();
            }
            return content;
        }

        @Override
        public String toString() {
            String entries = Arrays.stream(this.entries)
                .map(KvEntryWithStats::getName)
                .collect(Collectors.joining(", ", "[", "]"));
            return "{txn=" + txn + ", entries=" + entries + '}';
        }
    }
}
